#! /usr/bin/env python
# -*- coding: latin-1 -*-

# belltiming.py: Visualizing timing of Bell experiment data
# Copyright (C) 2003-2011 Jan-ke Larsson
# Released under GNU GPLv3, see below
#
# Created 2003 by Jan-ke Larsson <jan-ake.larsson@liu.se>
#
# Version 1.3 July 2013
# Changes: wxPython syntax
#
# Version 1.2 June 2011
# Changes: fix typo
#
# Version 1.1 November 2010
# Changes: adapt for numpy
#
# Version 1.0 June 2006
# Initial public release
#
# The program is intended to display experimental data in an easy way,
# understandable by non-specialists also.
#
# Display of: raw data on the top, different colors are different
# settings and different results (2x2=4 colors). On the bottom left
# are Bell correlations, where the boxes that show the correlations
# are five standard deviations high. On the bottom right is the
# following data: Bell inequality (apparent) violation, efficiency
# (that is, number of coincidences/number of single events) and
# finally number of coincidences.
#
# There are three sliders. The top one is for panning the data
# left-right, the mid one is for zooming, and the bottom one controls
# the width of the coincidence window used. The checkbox is for using
# fixed coincidence windows rather than a floating coincidence window.
#
# When you run this for the first time, you might try the following:
# pan left (by holding down the mousebutton in the tray to the left of
# the top slider) until you find an event that appears to have two
# coincident events. Now drag the middle slider to the left to zoom
# these events. And then drag the bottom slider to the left so that
# the coincidence window is narrow enough to have only two events in
# it. See the Bell correlations on the bottom rise as you do this. At
# a certain point you'll get a violation of the Bell inequality.
#
#
# The data file format is that of a zip file for each site containing
# a file with bigendian 64-bit floats that specify the time of an
# event and a file with 16-bit integers with the corresponding
# results, see the loaddata() function below. This data format is that
# used by Gregor Weihs et al in the Innsbruck 1998 experiment.
#
# The code is only sparsely documented since it never was meant for
# distribution. I have decided to release it under GPL in case there
# is anyone that finds it useful. If it breaks, you get to keep the
# pieces. But feel free to email me questions and suggestions for
# improvement.
#
# ####################################################################
#
# belltiming.py: Visualizing timing of Bell experiment data
# Copyright (C) 2003-2011 Jan-ke Larsson
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#
#
# ####################################################################

from __future__ import division
import numpy as np
import wx
from math import sqrt,cos,sin,log
from sys import byteorder,argv
from zipfile import ZipFile

xres=950
xtot=100000
yres=600
size=.01
eps=1e-11

m_time=5.0
w_time=0.001
d_time=0.00001
i_enable=0

myfont=("times", "20")

def loaddata():
  global a_times,b_times,a_data,b_data
  #canvas2.create_text(xres*21/32,(yres/3)*1/4,text=out,tag="corr",font=myfont)

  alice=ZipFile("alice.zip",'r')
  # Alice's detection times
  a_times=np.fromstring(alice.read("longdist35_V.dat"),dtype=">f8").astype(np.float64)
  # Alice's data
  # 0=vertical, no rotation
  # 1=horizontal, no rotation
  # 2=vertical, 45\degree rotation
  # 3=horizontal, 45\degree rotation
  a_data=np.fromstring(alice.read("longdist35_C.dat"),dtype=">u2").astype(np.int)
  alice.close()

  bob=ZipFile("bob.zip",'r')
  # Bob's detection times
  b_times=np.fromstring(bob.read("longdist35_V.dat"),dtype=">f8").astype(np.float64)
  # Bob's data
  # 0=vertical, no rotation
  # 1=horizontal, no rotation
  # 2=vertical, 45\degree rotation
  # 3=horizontal, 45\degree rotation
  b_data=np.fromstring(bob.read("longdist35_C.dat"),dtype=">u2").astype(np.int)
  bob.close()

def print_moj(out,str,a):
  for i in range(len(a)):
    if a[i]:
      out+=str
    else:
      out+="    "
  print out

def singles(a_data,b_data):
  a_single=[0,0,0,0]
  b_single=[0,0,0,0]
  for i in range(4):
    a_single[i]=len(np.nonzero(a_data==i)[0])
    b_single[i]=len(np.nonzero(b_data==i)[0])
  return(a_single,b_single)

def basic_coincidence():
  #print "Calculating coincidences"
  # Index is the index of Alice's data, everywhere but in b_times.
  #
  # Find the immediately following detections at Bob's site.
  # Only do \.../
  # If needed I'll add /... and ...\ starts and ends later.
  # An entry == len(b_times) here means no detection follows
  following=np.searchsorted(b_times,a_times)
  # Uniqify: At most one makes a pair with another
  i=np.nonzero(following[:-1]!=following[1:])[0]
  ai_f=i
  bi_f=following[ai_f]
  ai_p=i+1
  bi_p=following[ai_p]-1
  # At this point, bi_f contains Bob's index of the detection
  # following the ones that have the indices ai_f at Alice.
  #
  # Time difference calculation
  diff_f=b_times[bi_f]-a_times[ai_f]
  diff_p=a_times[ai_p]-b_times[bi_p]
  # Link notation below: \ a detection at Bob follows a detection at
  # Alice, / a detection at Bob precedes a detection at Alice.
  # Detect chains: a_chain /\ b_chain \/
  a_chain = ai_f[1:]==ai_p[:-1]
  b_chain = bi_p==bi_f
  a_f_smaller_p = diff_f[1:]<diff_p[:-1]
  b_p_smaller_f = diff_p<diff_f
  while len(np.nonzero(a_chain)[0]) or len(np.nonzero(b_chain)[0]):
    #print ".",
    # Chain /\/
    # If such a chain is found and the middle time is less
    # than the outer times, remove /-/
    #print_moj("  ","/\/ ", a_chain[:30]*b_chain[1:31])
    #print_moj("  ","/\/ ", a_chain[:30]*a_f_smaller_p[:30]*b_chain[1:31])
    #print_moj("  ","/\/ ", a_chain[:30]*a_f_smaller_p[:30]*b_chain[1:31]*(1-b_p_smaller_f[1:31]))
    i=np.nonzero(a_chain*a_f_smaller_p*b_chain[1:]*(1-b_p_smaller_f[1:]))[0]
    ai_p[i] = -1
    bi_p[i] = -1
    ai_p[i+1] = -1
    bi_p[i+1] = -1
    # Chain \/\
    # If such a chain is found and the middle time is less
    # than the outer times, remove \-\
    #print_moj("","\/\ ", a_chain[:30]*b_chain[:30])
    #print_moj("","\/\ ", a_chain[:30]*(1-a_f_smaller_p[:30])*b_chain[:30])
    #print_moj("","\/\ ", a_chain[:30]*(1-a_f_smaller_p[:30])*b_chain[:30]*b_p_smaller_f[:30])
    i=np.nonzero(a_chain*(1-a_f_smaller_p)*b_chain[:-1]*b_p_smaller_f[:-1])[0]
    ai_f[i] = -2
    bi_f[i] = -2
    ai_f[i+1] = -2
    bi_f[i+1] = -2
    # Chain /\-
    # If such a chain is found and the ending time is less
    # than the previous time, remove /--
    #print_moj("  ","/\- ", a_chain[:30]*(1-b_chain[1:31]))
    #print_moj("  ","/\- ", a_chain[:30]*a_f_smaller_p[:30]*(1-b_chain[1:31]))
    i=np.nonzero(a_chain*a_f_smaller_p*(1-b_chain[1:]))[0]
    ai_p[i] = -1
    bi_p[i] = -1
    # Chain \/-
    # If such a chain is found and the ending time is less
    # than the previous time, remove \--
    #print_moj("","\/- ", (1-a_chain[:30])*b_chain[:30])
    #print_moj("","\/- ", (1-a_chain[:30])*b_chain[:30]*b_p_smaller_f[:30])
    i=np.nonzero((1-a_chain)*b_chain[:-1]*b_p_smaller_f[:-1])[0]
    ai_f[i] = -2
    bi_f[i] = -2
    # Chain -\/
    # If such a chain is found and the starting time is less
    # than the following time, remove --/
    #print_moj("  ","-\/ ", (1-a_chain[:30])*b_chain[1:31])
    #print_moj("  ","-\/ ", (1-a_chain[:30])*b_chain[1:31]*(1-b_p_smaller_f[1:31]))
    i=np.nonzero((1-a_chain)*b_chain[1:]*(1-b_p_smaller_f[1:]))[0]
    ai_p[i+1] = -1
    bi_p[i+1] = -1
    # Chain -/\
    # If such a chain is found and the middle time is less
    # than the following time, remove --\
    #print_moj("","-/\ ", a_chain[:30]*(1-b_chain[:30]))
    #print_moj("","-/\ ", a_chain[:30]*(1-a_f_smaller_p[:30])*(1-b_chain[:30]))
    i=np.nonzero(a_chain*(1-a_f_smaller_p)*(1-b_chain[:-1]))[0]
    ai_f[i+1] = -2
    bi_f[i+1] = -2
    a_chain = ai_p[:-1]==ai_f[1:]
    b_chain = bi_p==bi_f
    #print "a_chain", a_chain
    #print "b_chain", b_chain
  #print
  i=np.nonzero(ai_f>0)[0]
  global a_pair,b_pair,deltas
  a_pair=-np.ones(a_data.shape,np.int)
  b_pair=-np.ones(b_data.shape,np.int)
  deltas=11*np.ones(a_data.shape,np.float64)
  a_pair[ai_f[i]]=bi_f[i]
  deltas[ai_f[i]]=diff_f[i]
  i=np.nonzero(ai_p>0)[0]
  a_pair[ai_p[i]]=bi_p[i]
  deltas[ai_p[i]]=diff_p[i]
  ############################################################################
  #print "Sorting coincidences"
  global i_d_sort,a_sort,b_sort,d_sort
  i_d_sort=deltas.argsort()
  d_sort=deltas[i_d_sort]
  a_sort=a_data[i_d_sort]
  b_sort=b_data[a_pair[i_d_sort]]
  ############################################################################
  global a_single,b_single
  #print "Calculating singles"
  (a_single,b_single)=singles(a_data,b_data)


def total(corr_m,a_single,b_single):
  sum=[0,0,0,0]
  tot=[0,0,0,0]
  sum[0]=-(corr_m[0][0]+corr_m[2][2]-corr_m[0][2]-corr_m[2][0])
  tot[0]=corr_m[0][0]+corr_m[2][2]+corr_m[0][2]+corr_m[2][0]
  sum[1]=-(corr_m[0][1]+corr_m[2][3]-corr_m[0][3]-corr_m[2][1])
  tot[1]=corr_m[0][1]+corr_m[2][3]+corr_m[0][3]+corr_m[2][1]
  sum[2]=-(corr_m[1][0]+corr_m[3][2]-corr_m[1][2]-corr_m[3][0])
  tot[2]=corr_m[1][0]+corr_m[3][2]+corr_m[1][2]+corr_m[3][0]
  sum[3]=-(corr_m[1][1]+corr_m[3][3]-corr_m[1][3]-corr_m[3][1])
  tot[3]=corr_m[1][1]+corr_m[3][3]+corr_m[1][3]+corr_m[3][1]
  corr=[0.0,0.0,0.0,0.0]
  stdev=[0.0,0.0,0.0,0.0]
  eta_21=1.0
  for i in [0,1,2,3]:
    try:
      corr[i]=sum[i]/tot[i]
    except:
      corr[i]=17.0 # Not a number, see draw_corr()
    try:
      stdev[i]=sqrt((tot[i]**2.0-sum[i]**2.0)/(tot[i]**2.0)/(tot[i]-1.0))
    except:
      stdev[i]=2.0
    try:
      eta_21=min(eta_21,
                 (corr_m[i][0]+corr_m[i][1]+corr_m[i][2]+corr_m[i][3])/a_single[i],
                 (corr_m[0][i]+corr_m[1][i]+corr_m[2][i]+corr_m[3][i])/b_single[i])
    except:
      eta_21=17.0 # Not a number, see draw_corr()
  return(corr,stdev,eta_21)


def slot_correlate():
  global corr,stdev,eta_21,N_tot
  global a_i,b_i
  # Find detections that fit in our window.
  if 10.0/d_time>2**30:
    a_slot = (a_times//d_time).astype(np.int64)
    b_slot = (b_times//d_time).astype(np.int64)
  else:
    a_slot = (a_times//d_time).astype(np.int32)
    b_slot = (b_times//d_time).astype(np.int32)
  if app.Pending():
    return 1
  # First detection only
  a_i = np.concatenate(([0],np.nonzero(a_slot[:-1]!=a_slot[1:])[0] + 1),1)
  a_i_slot=a_slot[a_i]
  if app.Pending():
    return 1
  # First detection only
  b_i = np.concatenate(([0],np.nonzero(b_slot[:-1]!=b_slot[1:])[0] + 1),1)
  b_i_slot=b_slot[b_i]
  if app.Pending():
    return 1
  # Count singles
  (a_slot_single,b_slot_single)=singles(a_data[a_i],b_data[b_i])
  if app.Pending():
    return 1
  # Find matches
  following=np.searchsorted(b_i_slot,a_i_slot)
  if app.Pending():
    return 1
  b_i_slot=np.concatenate((b_i_slot,[0]),1)
  equal=np.nonzero(b_i_slot[following]==a_i_slot)[0]
  if app.Pending():
    return 1
  a_i=a_i[equal]
  b_i=b_i[following[equal]]
  if app.Pending():
    return 1
  corr_slot=[[0,0,0,0],[0,0,0,0],[0,0,0,0],[0,0,0,0]]
  for i in [0,1,2,3]:
    for j in [0,1,2,3]:
      corr_slot[i][j]=len(np.nonzero((a_data[a_i]==i)*(b_data[b_i]==j))[0])
  (corr,stdev,eta_21)=total(corr_slot,a_slot_single,b_slot_single)
  N_tot=len(equal)
  return 0

corr_i=0
corr_m=[[0,0,0,0],[0,0,0,0],[0,0,0,0],[0,0,0,0]]

def correlate():
  global corr,stdev,eta_21,N_tot
  global corr_i,corr_m
  corr_j=corr_i
  corr_i=np.searchsorted(d_sort,d_time)
  if corr_i>corr_j:
    # add entries
    for i in [0,1,2,3]:
      for j in [0,1,2,3]:
	corr_m[i][j]+=len(np.nonzero((a_sort[corr_j:corr_i]==i)\
                                     *(b_sort[corr_j:corr_i]==j))[0])
  elif corr_i<corr_j:
    # remove entries
    for i in [0,1,2,3]:
      for j in [0,1,2,3]:
	corr_m[i][j]-=len(np.nonzero((a_sort[corr_i:corr_j]==i)\
                                     *(b_sort[corr_i:corr_j]==j))[0])
  (corr,stdev,eta_21)=total(corr_m,a_single,b_single)
  N_tot=corr_i



ID_ABOUT = 101
ID_EXIT  = 102
ID_SCROLL = 103
ID_SCALE = 104
ID_DELTA = 105
ID_FIXED = 106
ID_ONLOAD = 107
ID_MOREINFO = 108
ID_DELAY = 109
ID_ONFIRST = 110

def DrawAxes(dc,(xres,yres)):
  dc.SetPen(wx.BLACK_PEN)
  l_time=m_time-w_time/2
  r_time=m_time+w_time/2

  # at least twenty tickmarks
  ticks=1
  while (w_time/ticks)<20:
    ticks=ticks/10
  if w_time/ticks>=100:
    ticks=ticks*5
  if w_time/ticks>=40:
    ticks=ticks*2
  # scroll["resolution"]=ticks/5
  for i in range(int(l_time/ticks+1),int(r_time/ticks+1)):
    dc.DrawLine((i*ticks-l_time)/w_time*xres,yres*.30-1.25*size*xres,
                (i*ticks-l_time)/w_time*xres,yres*.30+1.25*size*xres+1)
    dc.DrawLine((i*ticks-l_time)/w_time*xres,yres*.70-1.25*size*xres,
                (i*ticks-l_time)/w_time*xres,yres*.70+1.25*size*xres+1)
  ticks=ticks*5
  for i in range(int(l_time/ticks)+1,int(r_time/ticks)+1):
    label=str(i*ticks)
    extent=dc.GetTextExtent(label)[0]
    dc.DrawLine((i*ticks-l_time)/w_time*xres,yres*.30-1.75*size*xres,
                (i*ticks-l_time)/w_time*xres,yres*.30+1.75*size*xres+1)
    dc.DrawLine((i*ticks-l_time)/w_time*xres,yres*.70-1.75*size*xres,
                (i*ticks-l_time)/w_time*xres,yres*.70+1.75*size*xres+1)
    dc.DrawText(label,
                (i*ticks-l_time)/w_time*xres-extent/2,yres*.30+2*size*xres)
    dc.DrawText(label,
                (i*ticks-l_time)/w_time*xres-extent/2,yres*.70+2*size*xres)
  # Axes
  dc.DrawLine(0,yres*.30,xres,yres*.30)
  dc.DrawLine(0,yres*.70,xres,yres*.70)

def DrawData(dc,(xres,yres)):
  l_time=m_time-w_time/2-w_time/50
  r_time=m_time+w_time/2+w_time/50
  width=images[0].GetWidth()
  height=images[0].GetHeight()
  for i in range(np.searchsorted(a_times,l_time),
                 np.searchsorted(a_times,r_time)):
    #dc.SetBrush(brush[a_data[i]])
    #dc.DrawCircle((a_times[i]-m_time)/w_time*xres+xres/2,.30*yres,size*xres)
    dc.DrawBitmap(images[a_data[i]],
                  (a_times[i]-m_time)/w_time*xres+xres/2-width/2,
                  .30*yres-height/2,
                  True)
  for i in range(np.searchsorted(b_times,l_time),
                 np.searchsorted(b_times,r_time)):
    #dc.SetBrush(brush[b_data[i]])
    #dc.DrawCircle((b_times[i]-l_time)/w_time*xres+xres/2,.70*yres,size*xres)
    dc.DrawBitmap(images[b_data[i]],
                  (b_times[i]-m_time)/w_time*xres+xres/2-width/2,
                  .70*yres-height/2,
                  True)

def DrawCorrAxes(dc,(xres,yres)):
  dc.SetPen(wx.BLACK_PEN)
  dc.SetBrush(wx.BLACK_BRUSH)
  dc.DrawLine(0,yres/2,xres*8/32+10,yres/2)#,arrow="last")
  dc.DrawPolygon(((xres*8/32+10,yres/2-3),
                  (xres*8/32+20,yres/2),
                  (xres*8/32+10,yres/2+3)))
  dc.DrawLine(xres*2/32,yres/2-size*xres,xres*2/32,yres/2+size*xres)
  dc.DrawLine(xres*4/32,yres-size*xres,xres*4/32,size*xres)#,arrow="last")
  dc.DrawPolygon(((xres*4/32-3,10),(xres*4/32,0),(xres*4/32+3,10)))
  dc.DrawLine(xres*4/32-size*xres,yres-20,xres*4/32+size*xres,yres-20)
  dc.DrawLine(xres*4/32-size*xres,20,xres*4/32+size*xres,20)
  dc.DrawLine(xres*6/32,yres/2-size*xres,xres*6/32,yres/2+size*xres)
  dc.DrawLine(xres*10/32,yres/2,xres*18/32+10,yres/2)#,arrow="last")
  dc.DrawPolygon(((xres*18/32+10,yres/2-3),
                  (xres*18/32+20,yres/2),
                  (xres*18/32+10,yres/2+3)))
  dc.DrawLine(xres*10/32,yres-size*xres,xres*10/32,size*xres)#,arrow="last")
  dc.DrawPolygon(((xres*10/32-3,10),(xres*10/32,0),(xres*10/32+3,10)))
  dc.DrawLine(xres*10/32-size*xres,yres-20,xres*10/32+size*xres,yres-20)
  dc.DrawLine(xres*10/32-size*xres,20,xres*10/32+size*xres,20)
  dc.DrawLine(xres*12/32,yres/2-size*xres,xres*12/32,yres/2+size*xres)
  dc.DrawLine(xres*16/32,yres/2-size*xres,xres*16/32,yres/2+size*xres)


def DrawDelta(dc,(xres,yres)):
  l_time=m_time-w_time/2
  r_time=m_time+w_time/2
  if i_enable:
    m_slot=m_time//d_time
    fraction=m_time/d_time-m_slot
    brush=[wx.Brush(wx.Colour(232-12*fraction,
                            231-13*fraction,
                            229-16*fraction)),
           wx.Brush(wx.Colour(255-13*fraction,
                            255-14*fraction,
                            255-16*fraction)),
           wx.WHITE_BRUSH,
           wx.WHITE_BRUSH,
           wx.WHITE_BRUSH,
           wx.Brush(wx.Colour(232+13*fraction,
                            231+14*fraction,
                            229+16*fraction)),
           wx.Brush(wx.Colour(220+12*fraction,
                            218+13*fraction,
                            213+16*fraction))]
    pen=[wx.Pen(wx.Colour(220,218,213)),
         wx.Pen(wx.Colour(110+110*fraction,
                        109+109*fraction,
                        107+106*fraction)),
         wx.Pen(wx.Colour(110*fraction,
                        109*fraction,
                        107*fraction)),
         wx.BLACK_PEN,
         wx.BLACK_PEN,
         wx.Pen(wx.Colour(110-110*fraction,
                        109-109*fraction,
                        107-107*fraction)),
         wx.Pen(wx.Colour(220-110*fraction,
                        218-109*fraction,
                        213-106*fraction))]
    for i in range(7):
      dc.SetBrush(brush[i])
      dc.SetPen(pen[i])
      j=int((m_slot*d_time+(i-3)*d_time-l_time)/w_time*xres)
      k=int((m_slot*d_time+(i-2)*d_time-l_time)/w_time*xres)
      dc.DrawRectangle(j,-1,k-j+1,yres+2)
    dc.SetPen(pen[0])
    dc.DrawLine((m_slot*d_time+4*d_time-l_time)/w_time*xres,-1,
                (m_slot*d_time+4*d_time-l_time)/w_time*xres,yres+2)
    # This one last so there'll be a black line for very narrow windows
    dc.SetPen(wx.BLACK_PEN)
    dc.DrawLine((m_slot*d_time-l_time)/w_time*xres,-1,
                (m_slot*d_time-l_time)/w_time*xres,yres+2)
  else:
    dc.SetBrush(wx.WHITE_BRUSH)
    dc.SetPen(wx.BLACK_PEN)
    if d_time>w_time:
      dc.DrawRectangle(-1,-1,xres+2,yres+2)
    elif int(d_time/w_time*xres)==0:
      dc.DrawRectangle(xres/2-1,-1,1,yres+2)
    else:
      dc.DrawRectangle((1-d_time/w_time)/2*xres,-1,
                       d_time/w_time*xres,yres+2)

def DrawCoinc(dc,(xres,yres)):
  dc.SetPen(wx.BLACK_PEN)
  l_time=m_time-w_time/2
  r_time=m_time+w_time/2
  y=.30*yres
  if i_enable:
    l_time_t=l_time//d_time*d_time
    r_time_t=(r_time//d_time+1)*d_time
    a_times_t=a_times[np.arange(np.searchsorted(a_times,l_time_t),
                                np.searchsorted(a_times,r_time_t))]
    a_slot=a_times_t//d_time
    b_times_t=b_times[np.arange(np.searchsorted(b_times,l_time_t),
                                np.searchsorted(b_times,r_time_t))]
    b_slot=b_times_t//d_time
    i=0
    j=0
    while i<len(a_slot) and j<len(b_slot):
      if a_slot[i]==b_slot[j]:
        dc.DrawLine((a_times_t[i]-l_time)/w_time*xres,y,
                    (b_times_t[j]-l_time)/w_time*xres,.70*yres)
        a_old=a_slot[i]
        while i<len(a_slot) and a_old==a_slot[i]:
          i=i+1
      elif a_slot[i]>b_slot[j]:
        j=j+1
      elif a_slot[i]<b_slot[j]:
        i=i+1
    ## Find detections that fit in our window.
    #if 10.0/d_time>2**30:
    #  a_slot = (a_times_t//d_time).astype(np.int64)
    #  b_slot = (b_times_t//d_time).astype(np.int64)
    #else:
    #  a_slot = (a_times_t//d_time).astype(np.int32)
    #  b_slot = (b_times_t//d_time).astype(np.int32)
    ## First detection only
    #a_i = concatenate(([0],np.nonzero(a_slot[:-1]!=a_slot[1:])[0] + 1),1)
    #a_i_slot=a_slot[a_i]
    ## First detection only
    #b_i = concatenate(([0],np.nonzero(b_slot[:-1]!=b_slot[1:])[0] + 1),1)
    #b_i_slot=b_slot[b_i]
    ## Find matches
    #following=np.searchsorted(b_i_slot,a_i_slot)
    #b_i_slot=concatenate((b_i_slot,[0]),1)
    #equal=np.nonzero(b_i_slot[following]==a_i_slot)[0]
    #a_i=a_i[equal]
    #b_i=b_i[following[equal]]
    #for i in range(len(a_i)):
    #  dc.DrawLine((a_times_t[a_i[i]]-l_time)/w_time*xres,y,
    #              (b_times_t[b_i[i]]-l_time)/w_time*xres,.70*yres)
  else:
    for i in range(np.searchsorted(a_times,l_time),np.searchsorted(a_times,r_time)):
      if a_pair[i]>=0 and deltas[i]<d_time:
        dc.DrawLine((a_times[i]-l_time)/w_time*xres,y,
                    (b_times[a_pair[i]]-l_time)/w_time*xres,.70*yres)

def DrawCorr(dc,(xres,yres)):
  dc.SetBrush(wx.GREEN_BRUSH)
  if corr[0]+corr[1]-corr[2]+corr[3]>2.0:
    dc.SetBrush(wx.RED_BRUSH)
  if corr[0]+corr[1]<10.0: # Not a number, see total()
    asize=(yres/2-20)*(corr[0]+corr[1])/sqrt(2)
    for i in range(50):
      dc.DrawLine(i*xres/200,yres/2-asize*sin(3.141526/50*i),
                  (1+i)*xres/200,yres/2-asize*sin(3.141526/50*(i+1)))
  if corr[2]+corr[3]<10.0: # Not a number, see total()
    bsize=(yres/2-20)*(-corr[2]+corr[3])/sqrt(2)
    for i in range(50):
      dc.DrawLine(xres*10/32+i*xres/200,yres/2-bsize*cos(3.141526/50*i),
                  xres*10/32+(1+i)*xres/200,yres/2-bsize*cos(3.141526/50*(i+1)))
  a=[2,6,16,12]
  b=[1,1,-1,1]
  for i in range(4):
    if corr[i]<10.0: # Not a number, see total()
      dc.DrawRectangle(xres*a[i]/32-size*xres,
                       yres/2-(yres/2-20)*(corr[i]+5*stdev[i]),
                       2*size*xres+1,
                       (yres/2-20)*(10*stdev[i])+1)
      dc.DrawLine(xres*a[i]/32-size*xres,yres/2-b[i]*(yres/2-20)/2,
                  xres*a[i]/32+size*xres+1,yres/2-b[i]*(yres/2-20)/2)

  normal=wx.Font(11,wx.ROMAN,wx.NORMAL,wx.NORMAL)
  sans=wx.Font(11,wx.SWISS,wx.NORMAL,wx.NORMAL)
  subscript=wx.Font(8,wx.ROMAN,wx.NORMAL,wx.NORMAL)
  dc.SetFont(normal)
  if corr[0]+corr[1]+corr[2]+corr[3]<10.0: # Not a number, see total()
    out="|E(AC)+E(AD)|+|E(BC)--E(BD)| = %.3f  (%.1f%%)" %\
         (corr[0]+corr[1]-corr[2]+corr[3],\
          (corr[0]+corr[1]-corr[2]+corr[3])/2/sqrt(2)*100)
  else:
    out="|E(AC)+E(AD)|+|E(BC)--E(BD)| = NaN"
  extent=dc.GetTextExtent(out)[0]
  # Set output text
  dc.DrawText(out,xres*24/32-extent/2,yres*1/4)
  # Extend absolute value braces
  dc.DrawText("|",xres*24/32-extent/2,yres*1/4-2)
  dc.DrawText("|",xres*24/32-extent/2
              +dc.GetTextExtent("|E(AC)+E(AD)")[0],yres*1/4-2)
  dc.DrawText("|",xres*24/32-extent/2
              +dc.GetTextExtent("|E(AC)+E(AD)|+")[0],yres*1/4-2)
  dc.DrawText("|",xres*24/32-extent/2
              +dc.GetTextExtent("|E(AC)+E(AD)|+|E(BC)--E(BD)")[0],yres*1/4-2)
  # Extend minus sign
  dc.DrawText("-",xres*24/32-extent/2
              +dc.GetTextExtent("|E(AC)+E(AD)|+|E(BC)-")[0]
              -dc.GetTextExtent("-")[0]/2,yres*1/4)
  dc.SetFont(sans)
  n=dc.GetTextExtent("n")[0]
  dc.SetFont(subscript)
  (subw,subh)=dc.GetTextExtent("2,1")
  dc.SetFont(normal)
  if eta_21>10.0: # Not a number, see total()
    out=" = NaN"
  elif eta_21>.11:
    out=" = %.1f%%" % (eta_21*100.0)
  elif eta_21>.011:
    out=" = %.2f%%" % (eta_21*100.0)
  elif eta_21>.0011:
    out=" = %.3f%%" % (eta_21*100.0)
  else:
    out=" = %.4f%%" % (eta_21*100.0)
  extent=dc.GetTextExtent(out)[0]
  dc.SetFont(sans)
  dc.DrawText("n",xres*24/32-(n+subw+extent)/2,yres*2/4)
  dc.SetFont(subscript)
  line=dc.GetTextExtent("|")[0]
  dc.DrawText("|",xres*24/32-(n+subw+extent)/2+n-line+1,yres*2/4+subh/2)
  dc.DrawText("2,1",xres*24/32-(n+subw+extent)/2+n,yres*2/4+subh/2)
  dc.SetFont(normal)
  dc.DrawText(out,xres*24/32-(n+subw+extent)/2+n+subw,yres*2/4)
  dc.SetFont(subscript)
  (subw,subh)=dc.GetTextExtent("tot")
  dc.SetFont(normal)
  n=dc.GetTextExtent("N")[0]
  out=" = %d" % N_tot
  extent=dc.GetTextExtent(out)[0]
  dc.DrawText("N",xres*24/32-(n+subw+extent)/2,yres*3/4)
  dc.SetFont(subscript)
  dc.DrawText("tot",xres*24/32-(n+subw+extent)/2+n,yres*3/4+subh/2)
  dc.SetFont(normal)
  dc.DrawText(out,xres*24/32-(n+subw+extent)/2+n+subw,yres*3/4)


class DataPanel(wx.Panel):
  def __init__(self,parent):
    wx.Panel.__init__(self,parent,-1)#,size=wx.Size(xres, yres))
    self.parent=parent
    self.SetStatusText=parent.SetStatusText
    self.onload=wx.Timer(self,ID_ONLOAD)
    self.moreinfo=wx.Timer(self,ID_MOREINFO)
    wx.EVT_PAINT(self, self.OnPaint)
    wx.EVT_SIZE(self, self.OnPaint)
    wx.EVT_TIMER(self, ID_ONLOAD, self.OnLoad)
    wx.EVT_TIMER(self, ID_MOREINFO, self.Moreinfo)
  def Moreinfo(self, event=None):
    a=len(a_times)
    b=len(b_times)
    self.SetStatusText("Alice has %d data points; Bob has %d data points."%
                       (a,b))
  def OnLoad(self, event=None):
    dc = wx.BufferedPaintDC( self )
    font = dc.GetFont()
    font.SetPointSize(9)
    dc.SetFont(font)
    dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
    dc.Clear()
    DrawDelta(dc,self.GetSize())
    DrawAxes(dc,self.GetSize())
    basic_coincidence()
    DrawCoinc(dc,self.GetSize())
    DrawData(dc,self.GetSize())
  def OnPaint(self, event=None):
    dc = wx.BufferedPaintDC( self )
    font = dc.GetFont()
    font.SetPointSize(9)
    dc.SetFont(font)
    dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
    dc.Clear()
    DrawDelta(dc,self.GetSize())
    DrawAxes(dc,self.GetSize())
    try:
      DrawCoinc(dc,self.GetSize())
      DrawData(dc,self.GetSize())
    except NameError:
      loaddata()
      try:
        DrawData(dc,self.GetSize())
        self.SetStatusText("Data from Weihs et al, PRL 81:5039 (1998), http://www.quantum.univie.ac.at/research/photonentangle/bellexp/data.html")
        self.onload.Start(1,wx.TIMER_ONE_SHOT)
        self.moreinfo.Start(20000,wx.TIMER_ONE_SHOT)
      except:
        pass

class CorrPanel(wx.Panel):
  def __init__(self,parent):
    wx.Panel.__init__(self,parent,-1)#,size=wx.Size(xres, yres))
    self.onfirst=wx.Timer(self,ID_ONFIRST)
    wx.EVT_PAINT(self, self.OnPaint)
    wx.EVT_SIZE(self, self.OnPaint)
    wx.EVT_TIMER(self, ID_ONFIRST, self.OnFirst)
  def OnFirst(self, event=None):
    dc = wx.BufferedPaintDC( self )
    dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
    dc.Clear()
    DrawCorrAxes(dc,self.GetSize())
    try:
      correlate()
      DrawCorr(dc,self.GetSize())
    except NameError:
      self.onfirst.Start(100,wx.TIMER_ONE_SHOT)
  def OnPaint(self, event=None):
    dc = wx.BufferedPaintDC( self )
    dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
    dc.Clear()
    DrawCorrAxes(dc,self.GetSize())
    try:
      DrawCorr(dc,self.GetSize())
    except NameError:
      self.onfirst.Start(100,wx.TIMER_ONE_SHOT)



class TimeScrollBar(wx.BoxSizer):
  class ScrollBar(wx.ScrollBar):
    """Sets the global m_time in correspondence with the position of the
    scrollbar"""
    def set(self,new):
      global m_time
      m_time=new
      self.SetThumbPosition(m_time/10.0*self.GetRange())
      self.canvas.OnPaint()
    def __init__(self,parent):
      wx.ScrollBar.__init__(self,parent,ID_SCROLL)
      # The number 2 is needed, 0 gives no page events, 1 gives page=line
      self.SetScrollbar(50000, 4000, 104000, 2)
      self.canvas=parent.canvas
      wx.EVT_SCROLL_THUMBTRACK(self,self.Thumb)
      wx.EVT_SCROLL_PAGEDOWN(self,self.Pagedown)
      wx.EVT_SCROLL_PAGEUP(self,self.Pageup)
      wx.EVT_SCROLL_LINEDOWN(self,self.Linedown)
      wx.EVT_SCROLL_LINEUP(self,self.Lineup)
    def Thumb(self,event=None):
      self.set(10.0*self.GetThumbPosition()/self.GetRange())
    def Pagedown(self,event=None):
      for i in range(15):
        self.set(m_time+0.03*w_time)
    def Pageup(self,event=None):
      for i in range(15):
        self.set(m_time-0.03*w_time)
    def Linedown(self,event=None):
      self.set(m_time+0.005*w_time)
    def Lineup(self,event=None):
      self.set(m_time-0.005*w_time)
  class TextPanel(wx.Panel):
    def __init__(self,parent,scrollbar):
      wx.Panel.__init__(self,parent,-1,size=wx.Size(xres, 16))
      self.scrollbar=scrollbar
      wx.EVT_SIZE(self, self.OnPaint)
      wx.EVT_PAINT(self, self.OnPaint)
    def OnPaint(self, event=None):
      dc = wx.PaintDC( self )
      dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
      dc.Clear()
      dc.SetPen(wx.Pen("BLACK",1))
      dc.DrawLine(0,0,xres,0)
      for i in range(11):
        text="%0.1f" % i
        dc.DrawText(text,i*(xres-28-int(.4/10.4*xres))/10+14
                    +int(.4/10.4*xres)/2-dc.GetTextExtent(text)[0]/2,0)
  def __init__(self,parent):
    wx.BoxSizer.__init__(self,wx.VERTICAL)
    self.scrollbar=self.ScrollBar(parent)
    self.textbox=self.TextPanel(parent,self.scrollbar)
    self.Add(self.textbox, 0, wx.EXPAND)
    self.Add(self.scrollbar, 0, wx.EXPAND)

class ScaleScrollBar(wx.BoxSizer):

  class TextPanel(wx.Panel):
    def __init__(self,parentwin,scrollbar):
      wx.Panel.__init__(self,parentwin,-1,size=wx.Size(xres, 18))
      self.scrollbar=scrollbar
      wx.EVT_PAINT(self, self.OnPaint)
      wx.EVT_SIZE(self, self.OnPaint)
    def OnPaint(self, event=None):
      dc = wx.BufferedPaintDC( self )
      dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
      dc.Clear()
      dc.SetPen(wx.BLACK_PEN)
      for i in range(8):
        dc.DrawLine((xres-32-self.scrollbar.width)/8*i
                    +16+self.scrollbar.width/2,14,
                    (xres-32-self.scrollbar.width)/8*i
                    +16+self.scrollbar.width/2,18)
        for j in range(2,10):
          dc.DrawLine((xres-32-self.scrollbar.width)/8*(i+log(j)/log(10))
                      +16+self.scrollbar.width/2,16,
                      (xres-32-self.scrollbar.width)/8*(i+log(j)/log(10))
                      +16+self.scrollbar.width/2,18)
      dc.DrawLine(xres-16-self.scrollbar.width/2,14,
                  xres-16-self.scrollbar.width/2,18)
      font = dc.GetFont()
      font.SetPointSize(9)
      dc.SetFont(font)
      text="%0.1e" % (10**(self.scrollbar.pos/100-9))
      dc.DrawText(text,(self.scrollbar.pos+self.scrollbar.width/2)
                  /(800+self.scrollbar.width)*(xres-32)
                  +16-dc.GetTextExtent(text)[0]/2,1)

  def set(self,new):
    global w_time
    w_time=10**(new/100-9.0)
    self.pos=new
    self.width=int(800*.04)
    self.scrollbar.SetScrollbar(self.pos, self.width, 800+self.width, 1)
    self.textbox.OnPaint()
  def Thumb(self,event=None):
    new=self.scrollbar.GetThumbPosition()
    self.set(new)
    self.canvas.OnPaint()
  def __init__(self,parent):
    wx.BoxSizer.__init__(self,wx.VERTICAL)
    self.canvas=parent.canvas
    self.scrollbar=wx.ScrollBar(parent,ID_SCALE)
    self.textbox=self.TextPanel(parent,self)
    self.set(-3.0*100+900)
    self.Add(self.textbox, 0, wx.EXPAND)
    self.Add(self.scrollbar, 0, wx.EXPAND)
    wx.EVT_SCROLL(self.scrollbar,self.Thumb)

class DeltaScrollBar(wx.BoxSizer):

  class TextPanel(wx.Panel):
    def __init__(self,parentwin,scrollbar):
      wx.Panel.__init__(self,parentwin,-1)
      self.scrollbar=scrollbar
      wx.EVT_SIZE(self, self.OnPaint)
      wx.EVT_PAINT(self, self.OnPaint)
    def OnPaint(self, event=None):
      dc = wx.BufferedPaintDC( self )
      dc.SetBackground(wx.Brush(wx.Colour(240,241,242)))
      dc.Clear()
      dc.SetPen(wx.BLACK_PEN)
      for i in range(8):
        dc.DrawLine((xres-32-self.scrollbar.width)/8*i
                    +16+self.scrollbar.width/2,14,
                    (xres-32-self.scrollbar.width)/8*i
                    +16+self.scrollbar.width/2,18)
        for j in range(2,10):
          dc.DrawLine((xres-32-self.scrollbar.width)/8*(i+log(j)/log(10))
                      +16+self.scrollbar.width/2,16,
                      (xres-32-self.scrollbar.width)/8*(i+log(j)/log(10))
                      +16+self.scrollbar.width/2,18)
      dc.DrawLine(xres-16-self.scrollbar.width/2,14,
                  xres-16-self.scrollbar.width/2,18)
      font = dc.GetFont()
      font.SetPointSize(9)
      dc.SetFont(font)
      text="%0.1e" % (10**(self.scrollbar.pos/100-9))
      dc.DrawText(text,(self.scrollbar.pos+self.scrollbar.width/2)
                  /(800+self.scrollbar.width)*(xres-32)
                  +16-dc.GetTextExtent(text)[0]/2,1)

  def set(self,new):
    global d_time
    d_time=10**(new/100-9.0)
    self.pos=new
    self.width=int(800*.04)
    self.scrollbar.SetScrollbar(self.pos, self.width, 800+self.width, 1)
    self.textbox.OnPaint()
  def Delay(self,event=None):
    self.SetStatusText("Calculating...")
    while slot_correlate():
      app.Yield()
    self.queue=0
    self.SetStatusText("")
    self.canvas2.OnPaint()
  def Thumb(self,event=None):
    new=self.scrollbar.GetThumbPosition()
    self.set(new)
    self.canvas.OnPaint()
    if i_enable:
      if self.queue==0:
        self.queue=1
        self.delay.Start(1,wx.TIMER_ONE_SHOT)
    else:
      correlate()
      self.SetStatusText("")
      self.canvas2.OnPaint()
  def Click(self,event=None):
    global i_enable
    i_enable=self.fixed.IsChecked()
    self.Thumb()
  def __init__(self,parent):
    wx.BoxSizer.__init__(self,wx.VERTICAL)
    self.canvas=parent.canvas
    self.canvas2=parent.canvas2
    self.SetStatusText=parent.SetStatusText
    self.scrollbar=wx.ScrollBar(parent)
    self.textbox=self.TextPanel(parent,self)
    self.fixed=wx.CheckBox(parent,ID_FIXED,"Fixed-position time-windows")
    self.pos=0
    self.queue=0
    self.set(-5.0*100+900)
    self.Add(self.textbox, 0, wx.EXPAND)
    self.Add(self.scrollbar, 0, wx.EXPAND)
    self.Add(self.fixed, 0, wx.ALIGN_CENTRE)
    self.scrollbar.Bind(wx.EVT_SCROLL, self.Thumb)
    self.fixed.Bind(wx.EVT_CHECKBOX, self.Click)
    self.delay=wx.Timer(self.scrollbar,ID_DELAY)
    self.scrollbar.Bind(wx.EVT_TIMER, self.Delay)

class MyFrame(wx.Frame):
  def __init__(self, parent, ID, title):
    wx.Frame.__init__(self, parent, ID, title,
                     wx.DefaultPosition, wx.Size(xres, yres))
    self.CreateStatusBar()
    self.SetStatusText("Copyright (C) 2003-2005 Jan-Ake Larsson")
    menu = wx.Menu()
    menu.Append(ID_ABOUT, "&About",
                "More information about this program")
    menu.AppendSeparator()
    menu.Append(ID_EXIT, "E&xit", "Terminate the program")
    menuBar = wx.MenuBar()
    menuBar.Append(menu, "&File");
    self.SetMenuBar(menuBar)
    wx.EVT_MENU(self, ID_ABOUT, self.OnAbout)
    wx.EVT_MENU(self, ID_EXIT,  self.TimeToQuit)
    self.Bind(wx.EVT_SIZE, self.OnResize)
    self.canvas=DataPanel(self)
    self.canvas2=CorrPanel(self)
    self.scroll=TimeScrollBar(self)
    self.scale=ScaleScrollBar(self)
    self.delta=DeltaScrollBar(self)
    box = wx.BoxSizer(wx.VERTICAL)
    box.Add(self.canvas, 1, wx.EXPAND)
    box.Add(self.scroll, 0, wx.EXPAND)
    box.Add(self.scale, 0, wx.EXPAND)
    box.Add(self.delta, 0, wx.EXPAND)
    box.Add(self.canvas2, 1, wx.EXPAND)
    self.SetAutoLayout(True)
    self.SetSizer(box)
    self.Layout()
  def OnAbout(self, event):
    a=a_times.size()
    b=b_times.size()
    dlg = wx.MessageDialog(self,
                          "This program calculates the correlation for\n"
                          "Bell experiment data for different settings\n"
                          "of the time-window parameter.\n\n"
                          "Demonstration written by and\n"
                          "Copyright (C) 2003-2010 Jan-Ake Larsson\n\n"
                          "Data from Weihs et al, PRL 81:5039 (1998)\n"
                          "http://www.quantum.univie.ac.at/research/photonentangle/bellexp/data.html\n"+
                          ("%d+%d data points"%(a,b)),
                          "About", wx.OK | wx.ICON_INFORMATION)
    dlg.ShowModal()
    dlg.Destroy()
  def TimeToQuit(self, event):
    self.Close(true)
  def OnResize(self,event=None):
    global xres,yres
    (xres,yres)=self.GetSize()
    self.Layout()


class MyApp(wx.App):
    def OnInit(self):
        global frame
        frame = MyFrame(None, -1, "Analysis of Bell experiment data")
        frame.Show(True)
        self.SetTopWindow(frame)
        return True

app = MyApp(0)

brush=[wx.RED_BRUSH,wx.BLUE_BRUSH,wx.GREEN_BRUSH,
       wx.Brush(wx.Color(255,255,0))]
images=[wx.Bitmap("RedSphere.png"),wx.Bitmap("BlueSphere.png"),
        wx.Bitmap("GreenSphere.png"),wx.Bitmap("YellowSphere.png")]

app.MainLoop()
