/*
 * A vector quantizer for texture compression
 *
 * Copyright (c) 2012 Jens Ogniewski, Information Coding Group,
 *                    Linköpings University, Sweden
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "readppm.h"
#include "vec_quant.h"
#include "bmp_writer.h"

#define USE_YUV       0
#define DO_REFINEMENT 1
#define USE_DEBLOCK   0
#define DEBLOCK_WRAPAROUND   0

int main(int argc, char **argv) {
  unsigned int g, h, j, k, width, height;
  unsigned char *i_image, *o_image, *min_vecs, *max_vecs, *colsel, *tmp_image;
  unsigned short *luma1, *luma2;
  double err=0.0, psnr, rmse, ssim=0.0;
  const double c1=(0.01*3*255)*(0.01*3*255);
  const double c2=(0.03*3*255)*(0.03*3*255);

  if (argc<2) {
    printf ("\n\nUsage: vq_demo inputfile\n\n");
    return 0;
  }
  tmp_image=(unsigned char *)readppm(argv[1], (int *)&width, (int *)&height);
  if (tmp_image==NULL) {
    printf ("\n\nError opening inputfile!\n\n");
    return -1;
  }
  i_image  =(unsigned char *)malloc(width*height*4);
  if (i_image==NULL) {
    printf ("\n\nError allocating memory!\n\n");
    return -1;
  }
  luma1    =(unsigned short *) malloc(width*height*sizeof(unsigned short)); 
  if (luma1==NULL) {
    printf ("\n\nError allocating memory!\n\n");
    return -1;
  }
  luma2    =(unsigned short *) malloc(width*height*sizeof(unsigned short)); 
  if (luma2==NULL) {
    printf ("\n\nError allocating memory!\n\n");
    return -1;
  }

  //convert to RGBA
  for(g=0;g<height;g++) {
    for(h=0;h<width;h++) {
      i_image[(h*width+g)*4]=tmp_image[((width-1-h)*width+g)*3];
      i_image[(h*width+g)*4+1]=tmp_image[((width-1-h)*width+g)*3+1];
      i_image[(h*width+g)*4+2]=tmp_image[((width-1-h)*width+g)*3+2];
      i_image[(h*width+g)*4+3]=0xff;
    }
  }

  //vector_quantization2_backup(i_image,width, height, &min_vecs, &max_vecs, &colsel);
  vector_quantization(i_image,width, height, &min_vecs, &max_vecs, &colsel, USE_YUV, DO_REFINEMENT, USE_DEBLOCK, DEBLOCK_WRAPAROUND );
  vector_quantization_decode(min_vecs, max_vecs, colsel, width, height, &o_image);

  //compute PSNR and RMSE
  for (g=0;g<width*height;g++) {
    err=err+(o_image[g*4]-i_image[g*4])*(o_image[g*4]-i_image[g*4]);
    err=err+(o_image[g*4+1]-i_image[g*4+1])*(o_image[g*4+1]-i_image[g*4+1]);
    err=err+(o_image[g*4+2]-i_image[g*4+2])*(o_image[g*4+2]-i_image[g*4+2]);
    luma1[g]=o_image[g*4]+o_image[g*4+1]+o_image[g*4+2];
    luma2[g]=i_image[g*4]+i_image[g*4+1]+i_image[g*4+2];
  }

  err=err/(width*height);
  rmse=sqrt((double)err);
  psnr=10*log10((255*255)/err);
  printf("\n\nRMSE:  %f\npsnr: %f\n\n",rmse,psnr);

  for(g=0;g<(height-7);g++) {
    for(h=0;h<(width-7);h++) {
      double mx=0.0, my=0.0, vx=0.0, vy=0.0, cvxy=0.0;
      for(j=0;j<8;j++) {
        for(k=0;k<8;k++) {
          mx=mx+luma1[(g+j)*width+h+k];
          my=my+luma2[(g+j)*width+h+k];
        }
      }
      mx=mx/64.0;
      my=my/64.0;
      for(j=0;j<8;j++) {
        for(k=0;k<8;k++) {
          vx=vx+((luma1[(g+j)*width+h+k]-mx)*(luma1[(g+j)*width+h+k]-mx));
          vy=vy+((luma2[(g+j)*width+h+k]-my)*(luma2[(g+j)*width+h+k]-my));
          cvxy=cvxy+((luma1[(g+j)*width+h+k]-mx)*(luma2[(g+j)*width+h+k]-my));
        }
      }
      vx=vx/64.0;
      vy=vy/64.0;
      cvxy=cvxy/64.0;
      ssim=ssim+((2*mx*my+c1)*(2*cvxy+c2)/(mx*mx+my*my+c1)/(vx+vy+c2));
    }
  }
  ssim=ssim/((width-7)*(height-7));
  printf("SSIM:  %f\n\n",ssim);

  write_bmp_image("vq_demo_out.bmp", o_image, width, height);

  free(tmp_image);
  free(i_image);
  free(min_vecs);
  free(max_vecs);
  free(colsel);
  free(o_image);

  return 0;
}
