/****************************************************************************
 * Simulation of cone-beam X-ray projections with 3D phantoms.              *
 *                                                                          *
 * author: Jens Mueller-Merbach                                             *
 ****************************************************************************/

/* Version 1.? */
/* Cleaning up the code was done by Jan Eriksson 1997(?) */

/* Version 2.0 */
/* Modified 970707 by Henrik Turbell, see printed documentation for details */
/* - cyndrical detectors */
/* - New, more logical (?), order when saving */
/* - Fixed bugs for non-square detector */

/* Version 2.1 */
/* Modified 990629 by Henrik Turbell */
/* - box primitive */
/* - voxelization parameters */

/* Version 3.0 */
/* Modified 2005-03 by Olle Seger and Maria Magnusson Seger */
/* - Matlab interface */
/* - Partly new structure of infiles */

/* Version 3.1 */
/* Modified 2005-11 -> by Maria Magnusson Seger */
/* - New updates, see //maja 2005 and the file READMEupdates */
/* - Fixed bugs,  see //maja 2005 and the file READMEbugs */

/*
The program take is licenced under the X11 license:

Copyright (C), 1996-2005, 
Jens Mueller-Merbach, Maria Magnusson, Henrik Turbell, Olle Seger

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>   
#include <stdlib.h>
#include <math.h>       
#include <errno.h>   
#include <malloc.h>       
#include <string.h>    
#include "take.h"

#ifdef STANDALONE
#define MALLOC malloc
#else
#define MALLOC mxMalloc
#endif

#ifndef STANDALONE
#include <mex.h>
#include <mat.h>
#include <matrix.h>

TAKEPAR *take(int argc, char **argv); /* prototype when mexing */
void voxelize(SETUP *su);       


/************************
 * The gateway function *
*************************/
void mexFunction( int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[] )
{ 
  int argc=2, i;
  int buflen,nx,ny,nz;
  int dims[3];
  char *argv[10];
  char name[128], *lineptr;
  double *pr, *pp;
  DETECTOR *pd;
  SETUP *su;
  TAKEPAR *pt;

  for (i=0; i<nrhs+2; i++)
    argv[i] = (char *) MALLOC(STR_LEN*sizeof(char));

  sprintf(argv[0], "%s", "take");
  sprintf(argv[1], "%s", "take.txt");

  if (nrhs > 0){
    for (i=0; i<nrhs; i++) {
      buflen = (mxGetM(prhs[i]) * mxGetN(prhs[i]) * sizeof(mxChar)) + 1;
      mxGetString(prhs[i], name, buflen);
      argv[i+2] = (char *) MALLOC(buflen*sizeof(char));
      sprintf(argv[i+2], "%s", name);
      if (strstr(name, "initfile")) {
        if ((lineptr = strchr(name,'=')) != NULL)   { 
          lineptr++;
          sscanf(lineptr, "%s", argv[1]);
        }
      }
    }
    argc=nrhs+2;
  }

  if(nlhs == 0){
    mexErrMsgTxt("Must have at least one output argument.");
  } 

  if(nlhs > 2){
    mexErrMsgTxt("Too many output arguments.");
  } 

  pt = take(argc, argv); 

  pd = pt->pdetector;
  su = pt->psetup;

  if (pd->proj != NULL) {
    nx = pd->pixel_x;
    ny = pd->pixel_y;
    nz = pd->nr_of_prj;
    pp = pd->proj;
  } else if (su->pvox != NULL) {
    nx = su->voxels[0];
    ny = su->voxels[1];
    nz = su->voxels[2];
    pp = su->pvox;
  } else
    mexErrMsgTxt("No data available.");

  if (nlhs>0) {
    dims[0]=nx;
    dims[1]=ny;
    dims[2]=nz;
    plhs[0] = mxCreateNumericArray(3, dims, mxDOUBLE_CLASS, mxREAL);
    /*    plhs[0] = mxCreateDoubleMatrix(nx, ny*nz, mxREAL);*/
    pr = mxGetPr(plhs[0]);
    for (i=0; i< nx*ny*nz; i++)
      *pr++ = *pp++;
  }

  if (nlhs>1) {
    if (pd->proj != NULL) {
      plhs[1] = mxCreateDoubleMatrix(1, 2, mxREAL);
      pr = mxGetPr(plhs[1]);
      pr[0] = pd->cylinder_radius; /* dist src <-> origo */
      pr[1] = 2*pd->fanangle/pd->pixel_x; /* angle between rays */
    } else if (su->pvox != NULL) {
      plhs[1] = mxCreateDoubleMatrix(1, 9, mxREAL);
      pr = mxGetPr(plhs[1]);
      pr[0] = su->voxlen[0];    /* side of voxelized volume */
      pr[1] = su->voxlen[1];
      pr[2] = su->voxlen[2];
      pr[3] = su->voxels[0];    /* nr of voxel */
      pr[4] = su->voxels[1];
      pr[5] = su->voxels[2];
      pr[6] = su->pos[0];	/* start voxel */
      pr[7] = su->pos[1];
      pr[8] = su->pos[2];
    } 
  }
}
#endif

/*************
* take setup *
**************/
#ifdef STANDALONE
main(int argc, char **argv)
#else
     TAKEPAR *take(int argc, char **argv)
#endif
{
  int prj, i;
  double *proj;

  /* Projection parameters */
  SETUP *su;            /* setup parameters */
  DETECTOR *pd;         /* detector parameters */
  TAKEPAR *pt;

  /* Allocate memory and initialize pointers */
  su = (SETUP *)  MALLOC(sizeof(SETUP));
  pd = (DETECTOR *) MALLOC(sizeof(DETECTOR));
  su->pp = (POLY *) MALLOC(sizeof(POLY));
  
  pt = (TAKEPAR *) MALLOC(sizeof(TAKEPAR));  
  pt->psetup = su;
  pt->pdetector = pd;

  /* Read initialization data: file names and some parameters */
  read_init(pt, argc, argv);

  /* Read phantom */
  su->po = read_phantom(pt);

  /* Read detector file */
  read_detector(pt);

  /* Read polychromatic parameters */
  beam_hard(pt);

  /* Read trajectory data */
  pd->to_trj = read_trj(pt);
  
  if (su->vox) {   /* voxelize the phantom */
    pd->proj = NULL;
    if (su->debug) printf("Voxelizing phantom as requested.\n");
    voxelize(su);
#ifdef STANDALONE
    return;
#else
    return(pt);
#endif
  } else {   /* or allocate memory for projection data */
    pd->proj = 
      (double *) MALLOC (pd->pixel_x*pd->pixel_y*pd->nr_of_prj*sizeof(double));
    if (pd->proj == NULL) 
      hejda("proj",1);
    su->pvox = NULL;
  }

  /*-------------------*
   * take projections  *
   *-------------------*/
  if (su->kind_of_att == IDEAL) { /* ideal rays */ //maja 2005-11
    for (prj=0; prj < pd->nr_of_prj; prj++) {
      projection_ideal(pt, prj);
    }
  } else {
    if (su->mono_i) {		/* moneenergetic rays */
      for (prj=0; prj < pd->nr_of_prj; prj++) {
	projection_mono(pt, prj);
      }
    } else {			/* polyenergetic rays */
      for (prj=0; prj < pd->nr_of_prj; prj++) {
	projection_poly(pt, prj);
      }
    }
  }
  proj = pd->proj;
  
  if (su->kind_of_att == TAKELOG) {
    for (i=0; i<pd->pixel_x*pd->pixel_y*pd->nr_of_prj; i++,proj++)
      *proj = log(su->pp->E0) - log(*proj);
  }
  
  /* Print the parameters */
  if (su->verbose) {
    print_log(pt);
  }

  /*------------------------------*
   * save projection data to file *
   *------------------------------*/
#ifdef STANDALONE
  {
    FILE   *fp;
    float  *pvolf0, *pvolf;
    double *pvold = pd->proj;
    int    i, isize;

    fp = fopen (su->pro_file, "wb");
    if (fp == NULL) 
      hejda(su->pro_file, 6);

    fwrite(&pd->pixel_x, sizeof(int), 1, fp);
    fwrite(&pd->pixel_y, sizeof(int), 1, fp);
    fwrite(&pd->nr_of_prj, sizeof(int), 1, fp);

    isize = pd->pixel_x*pd->pixel_y*pd->nr_of_prj;

    pvolf0 = (float *) MALLOC (isize*sizeof(float));
    if (pvolf0 == NULL) 
      hejda("proj",1);

    for (pvolf=pvolf0, i=0; i<isize; i++)
      *pvolf++ = (float) *pvold++;

    fwrite(pvolf0, sizeof(float), isize, fp);
    printf("Wrote projection data (%d x %d x %d) to file %s\n", 
	   pd->pixel_x, pd->pixel_y, pd->nr_of_prj, su->pro_file);

    fclose(fp);
  }
#endif

#ifdef STANDALONE
  return(0);
#else
  return(pt);
#endif
}


/******************************
 * take one projection, ideal *
 ******************************/
void projection_ideal(TAKEPAR *pt, int prj) //maja 2005-11
{
  int i;
  POINT pt_src = {0.0};		/* coordinates where ray starts at the source */
  POINT pt_det = {0.0};		/* coordinates where ray hits detector */
  POINT pt_src0 = {0.0};	/* coordinates where ray starts at the source */
  POINT pt_det0 = {0.0};	/* coordinates where ray hits detector */
  MATRIX D_notransl;		/* the detector matrix without translation */
  MATRIX T;			/* a transformation matrix */
  double raysum = 0.0;
  double rad = 0.0;		/* radiation deposited on one detector elem */
  int xpix, ypix;		/* counters for pixel_x, pixel_y */
  int xpoint, ypoint;		/* counters for points in pixel */
  int mirror, mpoint;		/* cast rays from detector sub-point to */
				/* corresponding source sub-point and */
				/* mirrored source sub-point */
  double leng = 0.0;		/* length of an intersection */
  int hit;	 		/* =0: no intersection, =1: intersection */
 
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;
  OBJECT *po = su->po;
  POLY *pp = su->pp;
  double *proj;			/* pointer into proj volume */
  TRAJECT *ptrj;

  /* set parameter pointers */
  proj = pd->proj + prj*pd->pixel_y*pd->pixel_x;
  ptrj = pd->to_trj + prj; 

  /* construct detector matrix without translation */  //maja 2005-11
  unit_matrix(D_notransl);
  mult_matrix(D_notransl, ptrj->detpos, D_notransl);
  D_notransl[0][3]=0.0;
  D_notransl[1][3]=0.0;
  D_notransl[2][3]=0.0;

  /* loop over detector elements */
  for (ypix = 0; ypix < pd->pixel_y; ypix++) {
    for (xpix = 0; xpix < pd->pixel_x; xpix++) {

      raysum = 0.0;
      
      /* loop over the source and 1 det. element  */
      for (ypoint = 0; ypoint < pd->points_xy; ypoint++) { 
	for (xpoint = 0; xpoint < pd->points_xy; xpoint++) {
	  for (mpoint = 0, mirror = 1; mpoint<pd->mi_nr; mpoint++, mirror*=-1) {

	    /* calculate a point on the source */
	    if (pd->mi_nr == 2) {
	      pt_src[0] = mirror * (pd->src_x_0 + xpoint * pd->src_x_delta);
	      pt_src[1] = 0.0;
	      pt_src[2] = mirror * (pd->src_y_0 + ypoint * pd->src_y_delta);
	    } else {
	      pt_src[0] = 0.0;
	      pt_src[1] = 0.0;
	      pt_src[2] = 0.0;
	    }

	    /* source point -> reference system */ //maja 2005-11
	    add_matrix(D_notransl, ptrj->srcpos, T);
	    trfm(pt_src, T, pt_src);
	    if (su->filter) {
	      add_matrix(D_notransl, pd->srcpos, T);
	      trfm(pt_src, T, pt_src0);
	    }

	    /* calculate a point on the detector */
	    calc_detector(pd, xpix, ypix, xpoint, ypoint, pt_det);

	    /* detector point -> reference system */
	    trfm(pt_det, ptrj->detpos, pt_det);
	    if (su->filter) {
	      trfm(pt_det, pd->detpos, pt_det0);
	    }

	    hit = 0;

	    /* loop over objects */
	    for(po=su->po; po; po=po->next) { 
	      if (po->phm) {
		leng = calc_intersection(po, pt_src, pt_det, su); //maja 2005-12
	      } else
		leng = calc_intersection(po, pt_src0, pt_det0, su); //maja 2005-12
	      if (leng>0) {
		po->nrhits++;
		raysum += leng * po->dens;
	      }
	    }

	  } /*** end mirror ***/
	} /*** end xpoint (src) ***/
      } /*** end ypoint (src) ***/

      *proj++ = raysum / (double)(pd->points_xy * pd->points_xy * pd->mi_nr);

    } /*** end xpix (det) ***/
  } /*** end ypix (det) ***/

}

/**************************************
/* take one projection, monoenergetic *
/**************************************/
void projection_mono(TAKEPAR *pt, int prj)
{
  int i;
  POINT pt_src = {0.0};		/* coordinates where ray starts at the source */
  POINT pt_det = {0.0};		/* coordinates where ray hits detector */
  POINT pt_src0 = {0.0};	/* coordinates where ray starts at the source */
  POINT pt_det0 = {0.0};	/* coordinates where ray hits detector */
  MATRIX D_notransl;		/* the detector matrix without translation */
  MATRIX T;			/* a transformation matrix */
  double raysum = 0.0;
  double rad = 0.0;		/* radiation deposited on one detector elem */
  int xpix, ypix;		/* counters for pixel_x, pixel_y */
  int xpoint, ypoint;		/* counters for points in pixel */
  int mirror, mpoint;		/* cast rays from detector sub-point to */
				/* corresponding source sub-point and */
				/* mirrored source sub-point */
  double leng = 0.0;		/* length of an intersection */
  int hit;	 		/* =0: no intersection, =1: intersection */
 
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;
  OBJECT *po = su->po;
  POLY *pp = su->pp;
  double *proj;			/* pointer into proj volume */
  TRAJECT *ptrj;

  /* mono initialization */
  i = su->mono_i;

  /* set parameter pointers */
  proj = pd->proj + prj*pd->pixel_y*pd->pixel_x;
  ptrj = pd->to_trj + prj; 

 /* construct detector matrix without translation */  //maja 2005-11
  unit_matrix(D_notransl);
  mult_matrix(D_notransl, ptrj->detpos, D_notransl);
  D_notransl[0][3]=0.0;
  D_notransl[1][3]=0.0;
  D_notransl[2][3]=0.0;

  /* loop over detector elements */
  for (ypix = 0; ypix < pd->pixel_y; ypix++) {
    for (xpix = 0; xpix < pd->pixel_x; xpix++) {

      raysum = 0.0;

      /* loop over the source and 1 det. element  */
      for (ypoint = 0; ypoint < pd->points_xy; ypoint++) { 
	for (xpoint = 0; xpoint < pd->points_xy; xpoint++) {
	  for (mpoint = 0, mirror = 1; mpoint<pd->mi_nr; mpoint++, mirror*=-1) {

	    /* calculate a point on the source */
	    if (pd->mi_nr == 2) {
	      pt_src[0] = mirror * (pd->src_x_0 + xpoint * pd->src_x_delta);
	      pt_src[1] = 0.0;
	      pt_src[2] = mirror * (pd->src_y_0 + ypoint * pd->src_y_delta);
	    } else {
	      pt_src[0] = 0.0;
	      pt_src[1] = 0.0;
	      pt_src[2] = 0.0;
	    }

	    /* source point -> reference system */ //maja 2005-11
	    add_matrix(D_notransl, ptrj->srcpos, T);
	    trfm(pt_src, T, pt_src);
	    if (su->filter) {
	      add_matrix(D_notransl, pd->srcpos, T);
	      trfm(pt_src, T, pt_src0);
	    }

	    /* calculate a point on the detector */
	    calc_detector(pd, xpix, ypix, xpoint, ypoint, pt_det);

	    /* detector point -> reference system */
	    trfm(pt_det, ptrj->detpos, pt_det);
	    if (su->filter) {
	      trfm(pt_det, pd->detpos, pt_det0);
	    }

	    hit = 0;

	    /* loop over objects */
	    for(po=su->po; po; po=po->next) { 
	      if (po->phm)
		leng = calc_intersection(po, pt_src, pt_det, su); //maja 2005-12
	      else
		leng = calc_intersection(po, pt_src0, pt_det0, su); //maja 2005-12
	      if (leng>0) {
		hit = 1;
		po->nrhits++;
		pp->mul[i] += leng * po->dens * po->mu[i];
	      }
	    }
	      
	    /* calc energy on detector */
	    if (hit) {		/* hit*/
	      raysum += pp->nrp[i] * pp->energy[i] * exp(-pp->mul[i]);
	      pp->mul[i] = 0.0;
	    } else {		/* no hit */
	      raysum += pp->nrp[i] * pp->energy[i];
	      pp->mul[i] = 0.0;
	    }

	  } /*** end mirror ***/
	} /*** end xpoint (src) ***/
      } /*** end ypoint (src) ***/

      *proj++ = raysum;
      
      if (raysum > pp->E0)
	pp->E0=raysum;

    } /*** end xpix (det) ***/
  } /*** end ypix (det) ***/

}

/**************************************
/* take one projection, polyenergetic *
/**************************************/
void projection_poly(TAKEPAR *pt, int prj)
{
  int i;
  POINT pt_src = {0.0};		/* coordinates where ray starts at the source */
  POINT pt_det = {0.0};		/* coordinates where ray hits detector */
  POINT pt_src0 = {0.0};	/* coordinates where ray starts at the source */
  POINT pt_det0 = {0.0};	/* coordinates where ray hits detector */
  MATRIX D_notransl;		/* the detector matrix without translation */
  MATRIX T;			/* a transformation matrix */
  double raysum = 0.0;
  double rad = 0.0;		/* radiation deposited on one detector elem */
  int xpix, ypix;		/* counters for pixel_x, pixel_y */
  int xpoint, ypoint;		/* counters for points in pixel */
  int mirror, mpoint;		/* cast rays from detector sub-point to */
				/* corresponding source sub-point and */
				/* mirrored source sub-point */
  double leng = 0.0;		/* length of an intersection */
  int hit;	 		/* =0: no intersection, =1: intersection */
 
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;
  OBJECT *po = su->po;
  POLY *pp = su->pp;
  double *proj;			/* pointer into proj volume */
  TRAJECT *ptrj;

  /*  double sigma2=0.0;*/		/* variance of actual ray */
  
  /* set parameter pointers */
  proj = pd->proj + prj*pd->pixel_y*pd->pixel_x;
  ptrj = pd->to_trj + prj; 

 /* construct detector matrix without translation */  //maja 2005-11
  unit_matrix(D_notransl);
  mult_matrix(D_notransl, ptrj->detpos, D_notransl);
  D_notransl[0][3]=0.0;
  D_notransl[1][3]=0.0;
  D_notransl[2][3]=0.0;

  /* loop over detector elements */
  for (ypix = 0; ypix < pd->pixel_y; ypix++) {
    for (xpix = 0; xpix < pd->pixel_x; xpix++) {
      
      raysum = 0.0;

      /*      sigma2 = 0.0;*/

      /* loop over the source and 1 det. element  */
      for (ypoint = 0; ypoint < pd->points_xy; ypoint++) { 
	for (xpoint = 0; xpoint < pd->points_xy; xpoint++) {
	  for (mpoint = 0, mirror = 1; mpoint<pd->mi_nr; mpoint++, mirror*=-1) {
	    if (pd->mi_nr == 2) {
	      pt_src[0] = mirror * (pd->src_x_0 + xpoint * pd->src_x_delta);
	      pt_src[1] = 0.0;
	      pt_src[2] = mirror * (pd->src_y_0 + ypoint * pd->src_y_delta);
	    } else {
	      pt_src[0] = 0.0;
	      pt_src[1] = 0.0;
	      pt_src[2] = 0.0;
	    }

	    /* source point -> reference system */ //maja 2005-11
	    add_matrix(D_notransl, ptrj->srcpos, T);
	    trfm(pt_src, T, pt_src);
	    if (su->filter) {
	      add_matrix(D_notransl, pd->srcpos, T);
	      trfm(pt_src, T, pt_src0);
	    }

	    /* calculate a point on the detector */
	    calc_detector(pd, xpix, ypix, xpoint, ypoint, pt_det);

	    /* detector point -> reference system */
	    trfm(pt_det, ptrj->detpos, pt_det);
	    if (su->filter) {
	      trfm(pt_det, pd->detpos, pt_det0);
	    }

	    hit = 0;

	    /* loop over objects */
	    for (po=su->po; po; po=po->next) { 
	      if (po->phm)
		leng = calc_intersection(po, pt_src, pt_det, su); //maja 2005-12
	      else
		leng = calc_intersection(po, pt_src0, pt_det0, su); //maja 2005-12
	      if (leng>0) {
		hit = 1;
		po->nrhits++;
		
		for (i=0; i<pp->lvls; i++) 
		  pp->mul[i] += leng * po->dens * po->mu[i];
	      }
	    }
	      
	    /* calc energy on detector */
	    if (hit) {		/* hit */
	      for (i=0; i < pp->lvls; i++) {
		raysum += pp->energy[i] * exp(-pp->mul[i]) * pp->nrp[i];
		/* sigma2 += 
		   pp->energy[i]*exp(-pp->mul[i])*pp->energy[i]*exp(-pp->mul[i])*pp->nrp[i];
		*/
		pp->mul[i] = 0.0;
	      }
	    } else {		/* no hit */
	      for (i=0; i < pp->lvls; i++) {
		raysum += pp->energy[i] * pp->nrp[i];
		/* sigma2 += pp->energy[i] * pp->energy[i] *pp->nrp[i]; */
		pp->mul[i] = 0.0;
	      }
	    }
	  
	  } /*** end mirror ***/
	} /*** end xpoint (src) ***/
      } /*** end ypoint (src) ***/
      
      if (su->noise) {
	// raysum += box_muller(raysum, sqrt(sigma2));
	/*	raysum += sigma2;*/
      }
      
      *proj++ = raysum;
      
      if (raysum > pp->E0)
	pp->E0=raysum;
      
    } /*** end xpix (det) ***/
  } /*** end ypix (det) ***/
  
}

/************************
 * voxelize the phantom *
/************************/
void voxelize(SETUP *su)        /* setup parameters */
{ 
  OBJECT *po = su->po;
  double *pv;
  int x1,y1,z1;			/* counters */
  int x2,y2,z2;			/* counters */
  int nx, ny, nz;
  double pos[3];      /* coordinates for the point (not the voxel!) */
  double dx, dy, dz;  /* distances between two points (not voxels!) */
		      /* +/- shows the direction of voxelizing      */
  int nrvox, suvox;
  double cuvox;
  double posx, posy, posz;
  double p_tf[3];		/* transformed point */

  suvox = su->voxpt;
  cuvox = (double) suvox*suvox*suvox;
  nrvox = su->voxels[0]*su->voxels[1]*su->voxels[2];
  su->pvox = (double *) MALLOC(nrvox*sizeof(double));

  if (su->pvox == NULL) 
    hejda("double *voxel",1);
  pv = su->pvox;

  nx = su->voxels[0];
  ny = su->voxels[1];
  nz = su->voxels[2];

  /* subvoxel sizes */
  dx = su->voxlen[0]/((double) (su->voxels[0]*suvox));
  dy = su->voxlen[1]/((double) (su->voxels[1]*suvox));
  dz = su->voxlen[2]/((double) (su->voxels[2]*suvox));

  /* start coordinates */
  su->pos[0]= posx = su->vox_center[0] - su->voxlen[0]/2.0 + dx/2.0;       
  su->pos[1]= posy = su->vox_center[1] - su->voxlen[1]/2.0 + dy/2.0;
  su->pos[2]= posz = su->vox_center[2] - su->voxlen[2]/2.0 + dz/2.0;


  if (su->debug) {
    printf("Nr of objects: %d\n", su->nr_obj);
    printf("%d * %d * %d voxels\n", nx, ny, nz);
    printf("Supersampling %d\n", suvox);
    printf("Voxel size  : dx=%f dy=%f dz=%f\n", dx, dy, dz);
    printf("Start position  : posx=%f posy=%f posz=%f\n", posx, posy, posz);
  }

  for (z1=0; z1 < nz; z1++) {                         /* z-loop */
    for (y1=0; y1 < ny; y1++) {                       /* y-loop */
      for (x1=0; x1 < nx; x1++) {                     /* x-loop */
	*pv = 0.0;
	for (z2=0; z2< suvox; z2++) {                        /* z2-loop */
	  pos[2] = posz + (z1*suvox+z2)*dz;
	  for (y2=0; y2< suvox; y2++) {                      /* y2-loop */
	    pos[1] = posy + (y1*suvox+y2)*dy;
	    for (x2=0; x2< suvox; x2++) {                    /* x2-loop */
              pos[0] = posx + (x1*suvox+x2)*dx;
	      /* loop over objects */
	      for(po=su->po; po; po=po->next) { 
		switch (po->kind_of_obj) {
		case 1: /* ellipsoid */
		  *pv += p_in_ellip(po, pos, p_tf);
		  break;
		case 2: /* cylinder */
		  *pv += p_in_cylind(po, pos, p_tf);
		  break;
		case 3: /* box */
		  *pv += p_in_box(po, pos, p_tf);
		  break;
		}
	      }
	    }
	  }
	}
	*pv /= cuvox;
	pv++;
      }
    }
  }
    
  if (su->debug) {
    printf("Final position: posx=%f posy=%f posz=%f\n", pos[0], pos[1], pos[2]);
  }
    
#ifdef STANDALONE
  {
    FILE   *fp;
    float  *pvolf0, *pvolf;
    double *pvold = su->pvox;
    int    i, isize;
  
    fp = fopen (su->vox_file, "wb");
    if (fp == NULL) 
      hejda(su->vox_file, 6);
 
    fwrite(su->voxels, sizeof(int), 3, fp);

    isize = su->voxels[0] * su->voxels[1] * su->voxels[2];

    printf("isize = %d\n", isize);

    pvolf0 = (float *) MALLOC (isize*sizeof(float));
    if (pvolf0 == NULL) 
      hejda("proj",1);

    for (pvolf=pvolf0, i=0; i<isize; i++)
      *pvolf++ = (float) *pvold++;
    
    fwrite(pvolf0, sizeof(float), isize, fp);
    printf("Wrote vox data (%d x %d x %d) to file %s\n", 
	   su->voxels[0], su->voxels[1], su->voxels[2], su->vox_file);

    fclose(fp);
  }
#endif
  
  return;
}
