%Optimisation on an implicit curve on the Weibull manifold
%Version 1.1

%INPUT
%t_init = the initial position of the camera's lens
%path = the path where the image focus stack is located
%lensMin = the minimum lens position. Usually 0
%lensMAX = the maximum lens position. Usually 1
%lensMinimumStepsize = the lens mechanical stepsize
%init_steplength = the initial step length (in terms of lens units) for the algorithm
%type = binary flag that chooses between two error functions. Set to 0 for Weib. Variance or 1 for Weib. eigen-orientation
%verbose = binary flag. If set to 1 then print output

%OUTPUT
%Ts =  An array of lens position where the optimiser sampled. The final entry in the list is where the sharp image is
%Error = An array of (negative) sharpness values. The final entry in the list is the optimum

% Copyright Vasileios Zografos vasileios@zografos.org
% Please cite: V.Zografos et al.  "The Weibull manifold in low-level image processing: An application to automatic image focusing", IMAVIS 31 (5), 2013, pp 401-417

function [Ts,Errors,CumIterations,MotorSteps]=WeibullOptimisation(t_init,path,lensMIN,lensMAX,lensMinimumStepsize,step_length,type,verbose)



%% PARAMETERS

F_TOL=1e-6;  %Error tolerance
T_TOL=lensMinimumStepsize/4;  %delta_t (lens steplength) tolerance
R_TOL=0.005; %Rao distance tolerance
S_TOL=0.01; % Weibull space step-length tolerance
delta_t=0.1; %the initial delta_t (lens steplength)
max_iterations=100; %quit after this many iterations
%system( ['.\C++\EDSDK++.exe ',num2str(-999)]); %initialise Canon camera
MotorSteps=0;
CurrentLensPos=0;

ALLTs=[];

sgn=sign(delta_t);
terminate=0;
MotorSteps=MotorSteps+round(abs(CurrentLensPos-t_init)/lensMinimumStepsize);
[F,x,y,CurrentLensPos]=SharpnessFunction(t_init,path,lensMIN,lensMAX,type,[0,0],CurrentLensPos);


ALLTs(end+1)=t_init;

if verbose
    fprintf('1, Error= %6.6f, (x,y)= %6.2f, %6.2f, t= %6.4f, step_length= %6.4f \n', F, x,y,t_init,step_length);
    figure(1000); hold on; plot(t_init,F,'rO');
end

Ts=t_init;
Errors=F;
Pos=[x,y];
Dx=0;
DTs=delta_t;


iterations=2;
CumIterations=1;

while ~terminate
    
    %Obtain new point
    t=Ts(iterations-1)+DTs(iterations-1);
     
    if t>lensMAX
        t=lensMAX;
    end
    if t<lensMIN
        t=lensMIN;
    end
    
       
    
    MotorSteps=MotorSteps+round(abs(CurrentLensPos-t)/lensMinimumStepsize);
    [F,x,y,CurrentLensPos]=SharpnessFunction(t,path,lensMIN,lensMAX,type,[x,y],CurrentLensPos);
   ALLTs(end+1)=t;
    
    
    CumIterations=CumIterations+1;
    
    if verbose
        fprintf('%i, Error= %6.6f, (x,y)= %6.4f, %6.4f, t= %6.4f, step_length= %6.4f \n',CumIterations, F, x,y,t,step_length);
        figure(1000); hold on; plot(t,F,'rO');
    end
    
    
    
    %check new point
    if F >= Errors(iterations-1)
        %Check region before x
        %take a nearby image 2 frames behind (to avoid noise)
        trial=2*sgn*lensMinimumStepsize;

        
        MotorSteps=MotorSteps+round(abs(CurrentLensPos-(Ts(iterations-1)-trial))/lensMinimumStepsize);
        [Ft,xt,yt,CurrentLensPos]=SharpnessFunction((Ts(iterations-1)-trial),path,lensMIN,lensMAX,type,[x,y],CurrentLensPos);
        ALLTs(end+1)=(Ts(iterations-1)-trial);
        CumIterations=CumIterations+1;
        
        if verbose
            fprintf('     TRIAL POINT %i, Error= %6.6f, (x,y)= %6.4f, %6.4f, t= %6.4f, step_length= %6.4f \n', CumIterations, Ft, xt,yt,Ts(iterations-1)-trial,step_length);
            figure(1000); hold on; plot(Ts(iterations-1)-trial,Ft,'rO');
        end
        
        
        if Ft<Errors(iterations-1)
            %overshoot with reduction. Global min at (x_{i-2},...,x_t]
            [x,y,step_length,F,t,delta_t,iter,ALLTs,CurrentLensPos,MotorStepsWolfe]=getWolfe(Ts(iterations-1)-trial,Ft,xt,yt,Pos(iterations-1,1),Pos(iterations-1,2),step_length,-sgn,path,lensMIN,lensMAX,lensMinimumStepsize,type,verbose,ALLTs,CurrentLensPos);
            sgn=sign(delta_t);

        else
            %overshoot without reduction. Global min at [x_{i-1},...,x_i)
            [x,y,step_length,F,t,delta_t,iter,ALLTs,CurrentLensPos,MotorStepsWolfe]=getWolfe(Ts(iterations-1),Errors(iterations-1),Pos(iterations-1,1),Pos(iterations-1,2),xt,yt,step_length,sgn,path,lensMIN,lensMAX,lensMinimumStepsize,type,verbose,ALLTs,CurrentLensPos);
            sgn=sign(delta_t);

        end
        MotorSteps=MotorSteps+MotorStepsWolfe;
        %add results to lists
        if F < Errors(iterations-1) %termination criteria
            
            Ts(iterations)=t;
            Errors(iterations)=F;
            Pos(iterations,:)=[x,y];
            DTs(iterations)=delta_t;
            
            iterations=iterations+1;
        else
            if verbose
                fprintf('No improvement \n');
            end
            terminate=1;
        end
        
        CumIterations=CumIterations+iter;
        
    else
        
        %termination criteria
        if verbose
            if abs(F - Errors(iterations-1)) <= F_TOL
                fprintf('F-TOL \n');
                terminate=1;
            end
            
            if Rao_dist([x,y],Pos(iterations-1,:),'Weibull') <= R_TOL
                fprintf('R-TOL \n');
                terminate=1;
            end
        end
        
        %Accept new point
        %and add it to the list
        Ts(iterations)=t;
        Errors(iterations)=F;
        Pos(iterations,:)=[x,y];
        
        
        %Estimate new point
        delta_x=estimatePoint(Pos(iterations-1,1),Pos(iterations-1,2),Pos(iterations,1),Pos(iterations,2),type);
        
        Dx(iterations)=delta_x;
        
        delta_t=sgn*(delta_x)*step_length;
        DTs(iterations)=delta_t;
        sgn=sign(delta_x)*sgn;

        
        iterations=iterations+1;
        
        
        %termination criteria
        if verbose
            if CumIterations==max_iterations
                fprintf('Max iterations \n');
                terminate=1;
            end
            
            if abs(delta_t)<=T_TOL
                fprintf('delta_t TOL \n');
                terminate=1;
            end
            
            if step_length <=S_TOL
                fprintf('Step length TOL \n');
                terminate=1;
                
            end
        end
        
        
        
        
    end
    
    

    
end

        
    

end

function [x,y,step_length,F,t,delta_t,iterations,ALLTs,CurrentLensPos,MotorSteps]=getWolfe(tcurr,F,x,y,x_prev,y_prev,step_length,sgn,path,lensMIN,lensMAX,lensMinimumStepsize,type,verbose,ALLTs,CurrentLensPos)
R_TOL=0.005; %Rao distance tolerance
S_TOL=0.01; %step-length tolerance
step_length=step_length/2;
MotorSteps=0;
R=inf;

iterations=0;
delta_t=0;
t=tcurr;
while step_length>S_TOL && R>R_TOL
    
    %Estimate new point
    delta_x=estimatePoint(x_prev,y_prev,x,y,type);
    
    delta_t=sgn*(delta_x)*step_length;
    
    
    t=tcurr+delta_t;
    
   MotorSteps=MotorSteps+round(abs(CurrentLensPos-t)/lensMinimumStepsize);
   [Fnew,xnew,ynew,CurrentLensPos]=SharpnessFunction(t,path,lensMIN,lensMAX,type,[x,y],CurrentLensPos);
    
    ALLTs(end+1)=t;
    
    R=Rao_dist([xnew,ynew],[x,y],'Weibull');
    if verbose
        fprintf('     WOLFE point Error= %6.4f, (x,y)= %6.4f, %6.4f, t= %6.4f, step_length= %6.4f \n',Fnew, xnew,ynew,t,step_length);
        figure(1000); hold on; plot(t,Fnew,'rO');
    end
    iterations=iterations+1;
    
    if Fnew<F
        %Wolfe point found
        F=Fnew;
        x=xnew;
        y=ynew;
        break;
        
    else
        %Reduce steplength and continue
        step_length=step_length/2;
    end
    
end



end


function delta_x=estimatePoint(x1,y1,x2,y2,type)
%estimate Delta_x for new point location. That is a vector with center (x2,y2) and direction from (x1,y1) towards (x2,y2).
%The length of the vector is delta_x


g=[x2-x1,y2-y1];
ng=norm(g);  g=g/ng;

F= W2Fisher(x2,y2);
frac=(g(2)*F(2,2)+g(1)*F(2,1))/(g(1)*F(1,1)+g(2)*F(1,2));

gG(2) = sqrt(1/(1+frac^2));
gG(1)=  -gG(2)*frac;

FG=-grad_func_eval(x2,y2,type);

sine_theta=dot(FG,g);
cosine_theta=dot(FG,gG);
tan_theta=sign(cosine_theta)*sine_theta/cosine_theta;

alpha=1;
delta_x=(2*alpha/(1+exp(-2*tan_theta))-alpha); %logistic kernel


end




