/*
The program take is licenced under the X11 license:

Copyright (C), 1996-2005, 
Jens Mueller-Merbach, Maria Magnusson, Henrik Turbell, Olle Seger

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

/*------------------------------------*
 * some numbers                       *
 *------------------------------------*/
#define MY_PI 3.1415926536
#define SMALL_NUMBER 0.000000001
#define BIG_NUMBER 1000000000

/*------------------------------------*
 * detector shapes                    *
 *------------------------------------*/
#define SHAPE_PLANAR 0
#define SHAPE_CYLINDRICAL_AROUND_SOURCE 1
#define SHAPE_CYLINDRICAL_AROUND_ROTATION_AXIS 2

/*------------------------------------*
 * object types                       *
 *------------------------------------*/
#define OBJECT_ELLIPSOID 1
#define OBJECT_CYLINDER 2
#define OBJECT_BOX 3
#define OBJECT_VOXELS 4

/*------------------------------------*
 * voxel volume projection generation * 
 *------------------------------------*/ 
//maja 2005-12
#define KOHLER 0
#define SIDDON 1
#define JOSEPH 2
#define JOSEPHNN 3

/*------------------------------------*
 * size of margin around voxel object *
 * to simplify border interpolation   *
 *------------------------------------*/
#define VOXMARG 2                    

/*------------------------------------*
 * attenuation types                  *
 *------------------------------------*/
#define TAKELOG 0
#define NOLOG 1
#define PHOTO_ATT 2
#define IDEAL 3

#define MONO_CHROME 0
#define POLY_CHROME 1

#define TRUE 1
#define FALSE 0

#define STR_LEN 120
#define NR_OF_MAT 20
#define MAX_LEVELS 512
#define DESC_FILES 9


/************************************/
/*   simple data types              */
/************************************/

/*----------------------*
 *  homogenous geometry *
 *----------------------*/
typedef double MATRIX[3][4];
typedef double POINT[3];

/*---------------*
 * lookup tables *
 *---------------*/
typedef struct table {
  double *pf;
  double *xvec, *yvec;
  int size_x, size_y;
} TABLE;


/************************************/
/*   complex data types             */
/************************************/
/*--------------------*
 * object information *
 *--------------------*/
typedef struct object {
    struct object *next;
    MATRIX obj_pos;
    MATRIX inv_obj;
    int nrhits; 		/* hit counter */
    int phm;			/* part of phantom? */
    double dens;		/* density */
    int kind_of_obj;            /* type of object */
    int mat;			/* material number */
    double *mu;			/* attenuation curve */
    float *voxeldata;		/* voxel volume data */
    int Nvox[3];		/* sixe of voxel volume without voxel margin */
} OBJECT;

/*-----------------------------------*
 * spectrum and material information *
 *-----------------------------------*/
typedef struct poly {
  int lvls;			/* number of energy levels */
  double *energy;		/* energy levels */
  double *nrp;		        /* Interpolated and normalized spectrum, */
				/* multiplied with nr_rays and nr_ph. */
  double E0;			/* energy in unattenuated ray */
  double *resp;			/* the detector response */
  double *mu;			/* interpolated mu / rho */
  double *mul;			/* mu(E) times length */
} POLY;

/*------------------------*
 * trajectory information *
 *------------------------*/
typedef struct traject {
  MATRIX detpos;
  POINT srcpos;
} TRAJECT;

/*----------------------*
 * detector information *
 *----------------------*/
typedef struct {
  double *proj;			/* the projection volume */
  double length_x, length_y;    /* length of detector in x-,y-direction */
  int pixel_x, pixel_y;		/* size of detector */
  int nr_of_prj;		/* nr of projections */
  int points_xy;
  double source_width, source_height; /* length of source in x-,y-direction */
  double src_x_0, src_y_0;      /* position of first sub sample on source */
  double src_x_delta, src_y_delta; /* distance between subsamples on source */
  double pix_lgth_x, pix_lgth_y; /* length of pixel in x-, y-direction */
  double fanangle;		/* max fanangle for cyl. det. */
  double skew;			/* tangens of the skew */
  double channel_offset;	/* value of detector horisontal offset */
				/*  use 0.25 for quarter offset */
  int shape;			/* shape of detector */
  double cylinder_radius;       /* distance between source and rotation axis */
  double source2det;            /* distance between source and rotation axis */
  int mi_nr;			/* nr of mirror points */
  int nr_rays;
  double initrot;		/* rotate phantom before taking projs */
  MATRIX R;			/* rotation matrix */
  POINT srcpos;			/* original source position */
  MATRIX detpos;		/* original detector position */
  TRAJECT *to_trj;              /* pointer to the trajectory array */
} DETECTOR;

/*---------------------*
 * general information *
 *---------------------*/
typedef struct {
  int verbose;
  int debug;
  char type[STR_LEN];		/* MATLAB or STANDALONE */
  char ini_file[STR_LEN];	/* name of init file */
  char ph_file[STR_LEN];	/* name of phantom file */
  char det_file[STR_LEN];	/* name of detector file */
  char pro_file[STR_LEN];	/* name of projection file */
  char log_file[STR_LEN];	/* name of log file */
  char trj_file[STR_LEN];	/* name of trajectory file */
  char vox_file[STR_LEN];	/* name of voxelization file */
  char scatter_file[STR_LEN];	/* name of scatter file */
  char spec_file[STR_LEN];      /* name of file that contains spectrum */
  char response_file[STR_LEN];  /* name of file that contains det. response */
  int nr_of_mat;		/* nr of materials */
  char *mu_file[NR_OF_MAT];	/* names of material files */
  int mat_flag[NR_OF_MAT];	/* is the mtrl used? */
  int spect;                    /* =0: spectrum file is lacking. */
				/* =1: spectrum file exists */
  int mat;                      /* =0: material file is lacking. */
				/* =1: material file exists */
  int vox;			/* do voxelization? */
  double *pvox;			/* pointer to voxel data */
  int voxels[3];		/* nr of voxels */
  double voxlen[3];		/* size of volume */
  double vox_center[3];		/* center of volume */
  double pos[3];		/* center of first voxel */
  int kind_of_att;		/* kind of attenuation modelling:           */
				/* if = 0: att = ln (sum i(e^att i))        */
				/* if = 1: att = sum i(e^att i)             */
				/* if = 2: same, but only photoelectr. eff. */
				/* if = 3: att = sum i (dens i)             */
  int kind_of_projgen;		/* kind of projection generation  method:   */ 
				//maja 2005-12
				/* if = 0: Kohler's method                  */
				/* if = 1: Siddon's method                  */
				/* if = 2: Joseph's method                  */
				/* if = 3: Joseph's method with nearest n.  */
  int voxpt;		        /* cubic root of number of points per voxel */
  int scatter;			/* =1: scatter, =0: no scatter */
  int noise;			/* =1: noise, =0: no noise */
  double nr_ph;			/* nr of photons */
  int filter;			/* filter ? */
  int nr_obj;			/* nr objects in the phantom */
  double mono;			/* monoenergetic energy */
  int mono_i;			/* its index */
  OBJECT *po;			/* pointer to the object array */
  POLY *pp;
} SETUP;

/*----------------------------------*
 * general and detector information *
 *----------------------------------*/
typedef struct {
    SETUP *psetup;
    DETECTOR *pdetector;
} TAKEPAR;

/**************************************************************/
/*                  prototypes                                */
/**************************************************************/
void projection_ideal(TAKEPAR *pt, int prj);
void projection_mono(TAKEPAR *pt, int prj);
void projection_poly(TAKEPAR *pt, int prj);
void read_init(TAKEPAR *pt, int no_args, char **args);
OBJECT *read_phantom (TAKEPAR *pt);
void read_detector (TAKEPAR *pt);          
TRAJECT *read_trj(TAKEPAR *pt);
void voxelize(SETUP *su);
float* read_volume_with_margin(SETUP *su, char*, int*);
void beam_hard(TAKEPAR *pt);
void interp(int n, double *yx, int m, double *z, double *w);
void interp1(int n, double *x, int sy, double *y, 
	      int m, double *z, int sw, double *w); 
void add_matrix (MATRIX A, POINT b, MATRIX C);
void unit_matrix (MATRIX A);
void mult_matrix (MATRIX m1, MATRIX m2, MATRIX m3);
void rot_matrix (char axis, double angle, MATRIX mrot);       
void inv_matrix (MATRIX m1, MATRIX m3);
void trfm(POINT point, MATRIX trans0, POINT point_tf);
void generate_mrx (OBJECT *po, 
		   double a, double b, double c,      
		   double x, double y, double z,      
		   double ang1, double ang2, double ang3, 
		   char ax1, char ax2, char ax3,       
		   char order);

double calc_intersection(OBJECT *po, POINT ray_src_tf, 
			 POINT ray_detector_tf, SETUP *su); //maja 2005-12
void calc_detector(DETECTOR *pd, int xpix, int ypix, int xpoint, 
		   int ypoint, POINT ray_detector);

char calc_intersection_ellip (POINT ray_source_tf,
			      POINT ray_detector_tf, 
			      POINT hit_pt1,
			      POINT hit_pt2);

char calc_intersection_cylind (POINT ray_source_tf, 
			       POINT ray_detector_tf, 
			       POINT hit_pt1, POINT hit_pt2);

char calc_intersection_box (POINT ray_source_tf, 
			    POINT ray_detector_tf, 
			    POINT hit_pt1,
			    POINT hit_pt2);

double p_in_ellip (OBJECT *, double *, double *); 
double p_in_cylind (OBJECT *, double *, double *);  
double p_in_box (OBJECT *, double *, double *);

//maja 2005-12
double lineint_in_voxvol_kohler(float*, int*, double*, double*);
double lineint_in_voxvol_siddon(float*, int*, double*, double*);
double lineint_in_voxvol_joseph(float*, int*, double*, double*);
double lineint_in_voxvol_josephnn(float*, int*, double*, double*);

double length (POINT p1, POINT p2, OBJECT *po);

void hejda (char *, int);
void print_log(TAKEPAR *pt);
char *my_fgets(char *line, FILE *fp);
double box_muller(double m, double s);
void loadmat(char *matfile, double *pd);
void load_single_event(TAKEPAR *pt);

double sum(double *x, int n);
double scp(double *x, double *y, int n);
void vmult(double *x, double*y, double *z, int n);
void svmult(double x, double*y, double *z, int n);

void bubblesort(int cnt, double *e_out);



