/*
The program take is licenced under the X11 license:

Copyright (C), 1996-2005, 
Jens Mueller-Merbach, Maria Magnusson, Henrik Turbell, Olle Seger

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include "take.h"


/***************************************/
/*  scalar  vector mult                */
/***************************************/
void svmult(double x, double*y, double *z, int n)
{
  int i;

  for (i=0; i<n; i++)
    z[i] = x*y[i];
}

/***************************************/
/*          vector mult                */
/***************************************/
void vmult(double *x, double*y, double *z, int n)
{
  int i;

  for (i=0; i<n; i++)
    z[i] = x[i]*y[i];
}

/***************************************/
/*          vector sum                 */
/***************************************/
double sum(double *x, int n)
{
  int i;
  double sum=0.0;

  for (i=0; i<n; i++)
    sum += x[i];

  return(sum);
}
/***************************************/
/*          scalar product             */
/***************************************/
double scp(double *x, double *y, int n)
{
  int i;
  double sum=0.0;

  for (i=0; i<n; i++)
    sum += x[i]*y[i];

  return(sum);
}

/***************************************/
/* square matrix*vector multiply y=A*x */
/***************************************/
void sq_mult_matrix(double *x, double *A, 
		    double *y, int n)
{
  int i;

  for (i=0; i<n; i++)
    y[i] = scp(x,A+i*n,n);
}

/***************************************/
/* square matrix transpose  (inplace)  */
/***************************************/
void sq_trans_matrix(double *x, int n)
{
  int i,j;
  double tmp;

  for (i=1; i<n; i++)
    for (j=i+1; j<n; j++) {
      tmp = x[i+j*n];
      x[i+j*n] = x[j+i*n];
      x[j+i*n] = tmp;
    }
}

/************************************************************/
/* interpolate vector yx, that is y(x), given on the form 
   [x(0) y(0) x(1) y(2) ... x(n-1) y(n-1)] 
   to a vector w = y(z) where
   w = [w(0) w(1) ... w(m-1)] and z = [z(0) z(1) ... z(m-1)] */
/************************************************************/
void interp(int n, double *yx, int m, double *z, double *w) 
{
  int i,j;
  double a,b;

  for(j=0; j<m; j++) {
    for(i=0; i<n-1; i++) {
      if((yx[2*i]<z[j]) && (z[j]<=yx[2*i+2])) {
	a = yx[2*i+2] - yx[2*i]; // increment
	b = (z[j] - yx[2*i])/a;	// weight coeff.
	w[j]=yx[2*i+1]*(1 - b) + yx[2*i+3]*b;
	continue;
      }
    }
  }
}

/************************************************************/
/* almost the same as interp, x and y are separate vectors  */ 
/* and with stride in y and w                               */
/************************************************************/
void interp1(int n, double *x, int sy, double *y, 
	      int m, double *z, int sw, double *w) 
{
  int i,j;
  double a,b;

  for(j=0; j<m; j++) {
    for(i=0; i<n-1; i++) {
      if((x[i]<z[j]) && (z[j]<=x[i+1])) {
	a = x[i+1] - x[i];                 // increment
	b = (z[j] - x[i])/a;	           // weight coeff.
	w[j*sw]=y[i*sy]*(1 - b) + y[(i+1)*sy]*b;
	continue;
      }
    }
  }
}

/***************************************/
/*            bubble sort              */
/***************************************/
void bubblesort(int n, double *x)
{
  double tmp;
  int i,j;

  for (i=0; i<n; i++) {
    for (j=0; j<n-i-1; j++) {
      if (x[j]>x[j+1]) {
	tmp = x[j];
	x[j] = x[j+1];
	x[j+1] = tmp;
      }
    }
  }
}

/**************************************/
/* add a translation to a hom. matrix */
/**************************************/
void add_matrix (MATRIX A, POINT b, MATRIX C) 
{
  int i,j;

  for (i=0; i<3; i++) {
    for (j=0; j<3; j++)
      C[i][j] = A[i][j];
    C[i][3] = A[i][3] + b[i];
  }
}

/***************************************/
/* initialize a hom. matrix to unity   */
/***************************************/
void unit_matrix (MATRIX A) 
{
  int i,j;

  for (i=0; i<3; i++) {
    for (j=0; j<4; j++) {
      if (i==j) A[i][j] = 1.0;
      else A[i][j] = 0.0;
    }
  }
}

/***************************************/
/* hom. matrix - vector multiply, y=Ax */
/***************************************/
void trfm(POINT x, MATRIX A, POINT y) 
{
  int i,j;       
  POINT z;

  for (i=0; i<3; i++) {
    z[i] = 0.0;
    for (j=0; j<3; j++) 
      z[i] += A[i][j] * x[j];
    z[i] += A[i][3];
  }

  for (i=0; i<3; i++)
    y[i] = z[i];
}

/*************************************/
/* multiply two hom. matrices , C=AB */
/*************************************/
void mult_matrix (MATRIX A, MATRIX B, MATRIX C)   
{
  int i,j,k;                  /* counters */
  MATRIX D;

  for (i=0; i<3; i++) {
    for (j=0; j<4; j++) {
      D[i][j] = 0.0;
      for (k=0; k<3; k++) {
	D[i][j] += A[i][k]*B[k][j];
      }
    }
    D[i][3] += A[i][3];
  }

  for (i=0; i<3; i++) 
    for (j=0; j<4; j++) 
      C[i][j] = D[i][j];
}

/****************************************************************************
 * generates a rotation matrix.                                             *
 * ------------------------------------------------------------------------ *
 * author: Jens Mueller-Merbach                                      nov 95 *
 ****************************************************************************
 * called by function: take.generate_mrx(), traj.detector_position(),       *
 * traj.expl_impl(), traj.safe_data(),
 ****************************************************************************
 * Input: axis. axis=1: x-axis. axis=2: y-axis. axis=3: z-axis.             *
 *        angle in degrees. Note: rotation always counter-clockwise         *
 * Output: rotation matrix mrot                                             *
 ****************************************************************************
 * Return: ./. (void)                                                       *
 ****************************************************************************/
void rot_matrix (char axis,              /* rotation axis */
		 double angle,            /* angle in degrees! */
		 MATRIX mrot)       /* rotation matrix */
{
  int i,j;            /* counters */
  double vcos;         /* value of cosinus(angle) */
  double vsin;         /* value of sinus(angle) */

  /*--------------------------------*
   * Initialize all elements        *
   *--------------------------------*/
  for (i = 0; i<3; i++)
    for (j = 0; j<4; j++)
      (i == j) ? (mrot[i][j] = 1.0) : (mrot[i][j] = 0.0);

  if ((angle < SMALL_NUMBER) && (angle > -SMALL_NUMBER)) return; 

  angle *= (MY_PI/180.0);
  vcos = cos(angle);
  vsin = sin(angle);

  if (axis == 1) {
    mrot[1][1] = mrot[2][2] = vcos;
    mrot[1][2] = -vsin;
    mrot[2][1] = vsin;
  } else if (axis == 2) {
    mrot[0][0] = mrot[2][2] = vcos;
    mrot[0][2] = vsin;
    mrot[2][0] = -vsin;
  } else if (axis == 3) {
    mrot[0][0] = mrot[1][1] = vcos;
    mrot[0][1] = -vsin;
    mrot[1][0] = vsin;
  } else 
    hejda("axis",3);

}

void inv_matrix (MATRIX m1,       /* input matrix, to be inverted */
		 MATRIX m3)       /* inverted matrix */
/****************************************************************************
 * invertes a 4x4 matrix which has [0 0 0 1] in its last row.               *
 * With such kind of last row the inversion is much simpler as for normal   *
 * (or general) 4x4 matrices.                                               *
 * ------------------------------------------------------------------------ *
 * author: Jens Mueller-Merbach                                      nov 95 *
 ****************************************************************************
 * called by function: take.generate_mrx()                                  *
 ****************************************************************************
 * Input: m1                                                                *
 * Output: m3 = (m1)^-1                                                     *
 ****************************************************************************
 * Return: ./. (void)                                                       *
 ****************************************************************************/
{
  double det = 0.0;		/* determinant */
  int i,j;			/* counter */

  /*-----------------------*
   * calculate determinant *
   *-----------------------*/
  for (i = 0; i<3; i++) {
    det += m1[0][i]*m1[1][(i==2) ? 0 : (i+1)]*m1[2][(i==0) ? 2 : (i-1)];
    det -= m1[2][i]*m1[1][(i==2) ? 0 : (i+1)]*m1[0][(i==0) ? 2 : (i-1)];
  }

  /*-----------------------*
   * check for singularity *
   *-----------------------*/
  if ((det < SMALL_NUMBER) && (det > -SMALL_NUMBER)) hejda("inv_matrix",2);

  /*------------------------------------*
   * calculate elements m00 through m22 *
   *------------------------------------*/
  for (i = 0; i<3; i++)
    for (j = 0; j<3; j++) {
      m3[i][j] = ( ( ((i+j)!=1) && ((i+j)!=3) )? 1.0 : (-1.0)) * 
	(m1[(j==0)? 1 : 0][(i==0)? 1 : 0] *
	 m1[(j==2)? 1 : 2][(i==2)? 1 : 2] -
	 m1[(j==0)? 2 : (j-1)][(j==1)?
			      ((i==2)? 1 : 2) : ((i==0)? 1 : 0)] *
	 m1[(j==2)? 0 : (j+1)][(j==1)?
			      ((i==0)? 1 : 0) : ((i==2)? 1 : 2)]);
      m3[i][j] /= det;
    }

  /*------------------------------------*
   * calculate elements m03 through m23 *
   *------------------------------------*/
  for (i = 0; i<3; i++) {
    m3[i][3] = ((i==1)? 1.0 : (-1.0)) *
      (m1[0][(i==0)? 1 : 0]*m1[1][(i==2)? 1 : 2]*m1[2][3] +     
       m1[0][(i==2)? 1 : 2]*m1[1][3]*m1[2][(i==0)? 1 : 0] +
       m1[0][3]*m1[1][(i==0)? 1 : 0]*m1[2][(i==2)? 1 : 2] -
       m1[0][3]*m1[1][(i==2)? 1 : 2]*m1[2][(i==0)? 1 : 0] -
       m1[0][(i==0)? 1 : 0]*m1[1][3]*m1[2][(i==2)? 1 : 2] -
       m1[0][(i==2)? 1 : 2]*m1[1][(i==0)? 1 : 0]*m1[2][3]);
    m3[i][3] /= det;
  }
}

/****************************************************************************
 * generates an object matrix (scaling, rotation, translation relative to   *
 * the reference base) based on the object data read from file or from      *
 * default. Also generates the inverse matrix.                              *
 * ------------------------------------------------------------------------ *
 * author: Jens Mueller-Merbach                                      dec 95 *
 ****************************************************************************
 * Input: object parameters                                                 *
 * Output: object matrix and inverse (contained both in to_obj)             *
 ****************************************************************************
 * Return: ./. (void)                                                       *
 ****************************************************************************/
void generate_mrx (OBJECT *po,                       /* object pointer */
		   double a, double b, double c,            /* radii */
		   double x, double y, double z,            /* translations */
		   double ang1, double ang2, double ang3,   /* angles */
		   char ax1, char ax2, char ax3,         /* rotational axes */
		   char order)                           /* order of rot. and trl. */
{
  MATRIX scl;          /* some matrices */
  MATRIX trl;
  MATRIX rot1; 
  MATRIX rot2;
  MATRIX rot3;
  MATRIX dummy1;
  MATRIX dummy2;
  MATRIX dummy3;
  int i,j;               /* counter */
  int nr_of_rot;      /* number of rotations */

  nr_of_rot = 3;
    
  /*---------------------------------------------*
   * initialize scaling and translation matrices *
   *---------------------------------------------*/
  for (i = 0; i<3; i++)
    for (j = 0; j<4; j++) {
      scl[i][j] = 0.0;
      trl[i][j] = 0.0;
    }
   trl[0][0] = 1.0;
   trl[1][1] = 1.0;
   trl[2][2] = 1.0;

  /*---------------------------------------------*
   * generate scaling and translation matrices   *
   *---------------------------------------------*/
  scl[0][0] = a;
  scl[1][1] = b;
  scl[2][2] = c;
  trl[0][3] = x;
  trl[1][3] = y;
  trl[2][3] = z;

  /*---------------------------------------------*
   * generate rotation matrices                  *
   * check first if rotation is necessary        *
   *---------------------------------------------*/
  if (ax3 == 0) nr_of_rot--;
  else rot_matrix(ax3, ang3, rot3);

  if ((ax2 == 0) &&(nr_of_rot == 2)) nr_of_rot--;
  else rot_matrix(ax2, ang2, rot2);

  if ((ax1 == 0) &&(nr_of_rot == 1)) nr_of_rot--;
  else rot_matrix(ax1, ang1, rot1);

  /*----------------------------------------------*
   * calculate transformation matrix according to *
   * transf = transl * rot1 * rot2 * rot3 * scale *
   * check if rotation matrix exists              *
   *----------------------------------------------*/
  if (order==0) {
    switch (nr_of_rot) {
    case 3:
      mult_matrix(rot2, rot3, dummy3);
      mult_matrix(rot1, dummy3, dummy2);
      mult_matrix(trl, dummy2, dummy1);
      mult_matrix(dummy1, scl, dummy3);
      break;
    case 2:
      mult_matrix(rot1, rot2, dummy2);
      mult_matrix(trl, dummy2, dummy1);
      mult_matrix(dummy1, scl, dummy3);
      break;
    case 1:
      mult_matrix(trl, rot1, dummy1);
      mult_matrix(dummy1, scl, dummy3);
      break;
    case 0:
      mult_matrix(trl, scl, dummy3);
      break;
    default:
      hejda("nr_of_rot",5);
    }
  } else {
    switch (nr_of_rot) {
    case 3:
      mult_matrix(rot2, rot3, dummy3);
      mult_matrix(rot1, dummy3, dummy2);
      mult_matrix(dummy2, trl, dummy1);
      mult_matrix(dummy1, scl, dummy3);
      break;
    case 2:
      mult_matrix(rot1, rot2, dummy2);
      mult_matrix(dummy2, trl, dummy1);
      mult_matrix(dummy1, scl, dummy3);
      break;
    case 1:
      mult_matrix(rot1, trl, dummy1);
      mult_matrix(dummy1, scl, dummy3);
      break;
    case 0:
      mult_matrix(trl, scl, dummy3);
      break;
    default:
      hejda("nr_of_rot",5);
    }
  }

  /*----------------*
   * invert matrix  *
   *----------------*/
  inv_matrix(dummy3, dummy1);

  /*-------------------------*
   * save matrices in po     *
   *-------------------------*/
  for (i = 0; i<3; i++) {
    for (j = 0; j<4; j++) {
      po->obj_pos[i][j] = dummy3[i][j];
      po->inv_obj[i][j] = dummy1[i][j];
    }
  }
} 



