/****************************************************************************
 * The input/output functions                                               *
 *                                                                          *
 * Written by Jens Mueller-Merbach, 1996-03                                 *
 * Updated by Olle Seger, 2003-2005                                         *
 * Updated by Maria Magnusson Seger, 2005-11                                *
 ****************************************************************************/

/*
The program take is licenced under the X11 license:

Copyright (C), 1996-2005, 
Jens Mueller-Merbach, Maria Magnusson, Henrik Turbell, Olle Seger

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>        
#include <stdlib.h>
#include <math.h>         
#include <errno.h>        
#include <malloc.h>       
#include <string.h>  
#include "take.h"

#ifdef STANDALONE //maja 2005-12
#define MALLOC malloc
#define CALLOC calloc
#else
#define MALLOC mxMalloc
#define CALLOC mxCalloc
#endif

#ifndef STANDALONE
#include <mex.h>
#endif

void read_command_init(TAKEPAR *pt, int argc, char **argv);


/****************************************************************************
 * read a line, but skip comment lines                                      *
 ****************************************************************************/
char *my_fgets(char *line, FILE *fp)
{
  char *buf;

  do {
    buf=fgets(line, STR_LEN, fp);
    if (buf==NULL) break;
  } while (*buf=='#');

  return(buf);
}

/****************************************************************************
 * reads some initialization data: names of phantom file, detector file,    *
 * trajectory file, projection file or if default values should be used.    *
 * It furthermore reads the kind of attenuation: linear (0) or log (1).     *
 * It also checks if the directories for the output files exist.            *
 *                                                                          *
 * author: Jens Mueller-Merbach                                      jan 96 *
 *                                                                          *
 * Input: ./.                                                               *
 * Output: names of files                                                   *
 * Return: array of 0 and 1. If 1: default, if 0: load file                 *
 * element[0] = ph_file, element[1] = det_file, element[2] = trj_file,      *
 * element[3] = pro_file, element[4] = voxelization                         *
 * defaul[5]=1: polychromatic source, defaul[6]=1: default energy spectrum, *
 * defaul[7]=1: default attenuation array                                   *
 * defaul[8]=1: simulate scatter                                            *
 *                                                                          *
 * There are now four types of attenuation: log, nolog, photo, ideal.       *
 * Maria Magusson Seger                                            Nov 2005 *
 ****************************************************************************/
void read_init(TAKEPAR *pt, int argc, char **argv)
{
  FILE *fp;
  int i;
  char line[STR_LEN], *lineptr;
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;

  sprintf(su->ini_file, "take.txt");
  
  if (argc > 1) {
    sscanf(argv[1], "%s", su->ini_file);
  }

  fp = fopen (su->ini_file, "rt");
  if (fp == NULL) {
    hejda("Init file ", 4);
  }

  /*----------------*
   * Default values *
   *----------------*/
  su->kind_of_att = NOLOG;
  su->kind_of_projgen = KOHLER; //maja 2005-12
  su->scatter=0;
  su->filter=0;			//maja 2005-11
  su->voxpt=1;
  su->verbose = 0;
  su->debug = 0;
  su->vox = 0;
  su->noise = 0;
  su->nr_ph = 1;
  su->nr_of_mat = 0;
  su->nr_obj = 0;
  su->po = NULL;
  su->pvox = NULL;
  pd->initrot = 0.0;
  su->mono = 0.0;
  su->mono_i = 0;
  su->spect = 0;
  su->mat = 0;			//maja 2005-11
  su->voxels[0] = su->voxels[1] = su->voxels[2] = 0; //maja 2005-11
  su->voxlen[0] = su->voxlen[1] = su->voxlen[2] = 0.0; //maja 2005-11
  su->vox_center[0] = su->vox_center[1] = su->vox_center[2] =0.0; //maja 2005-11


  for (i=0; i<NR_OF_MAT; i++) {
    su->mu_file[i] = (char *) CALLOC(STR_LEN, sizeof(char));
    su->mu_file[i][0] = '\0';
    su->mat_flag[i] = 0;
  }

  for (i=0; i<3; i++) su->vox_center[i] = 0.0;

  /*-----------------------------------*
   * default names for parameter files *
   *-----------------------------------*/
  sprintf(su->ph_file,"phm.txt");
  sprintf(su->det_file,"det.txt");
  sprintf(su->scatter_file,"");
  sprintf(su->trj_file,"trj.txt");
  sprintf(su->pro_file,"proj.dat");
  sprintf(su->log_file,"log.txt");
  sprintf(su->vox_file,"vox.dat");
  sprintf(su->spec_file,"");
  sprintf(su->response_file,"");

  /*----------------------------------------------------------*
   * read names of files and other initialization             *
   *----------------------------------------------------------*/
  while(my_fgets(line, fp) != NULL) {
    if ((lineptr = strchr(line,'=')) != NULL)   { 
      lineptr++;
      if (strstr(line,"verbose") != NULL) {
	su->verbose = 1;
      } else if (strstr(line,"phantom") != NULL) {
	sscanf(lineptr,"%s",su->ph_file);
      } else if (strstr(line,"detector") != NULL) {
	sscanf(lineptr,"%s",su->det_file);
      } else if (strstr(line,"scatter") != NULL) {
	sscanf(lineptr,"%s",su->scatter_file);
	su->scatter = 1;
      } else if (strstr(line,"trajectory") != NULL) {
	sscanf(lineptr,"%s",su->trj_file);
      } else if (strstr(line,"projection") != NULL) {
	sscanf(lineptr,"%s",su->pro_file);
      } else if (strstr(line,"attenuation") != NULL) {
	if (strstr(line,"log") != NULL)   su->kind_of_att = TAKELOG;
	if (strstr(line,"nolog") != NULL) su->kind_of_att = NOLOG;
	if (strstr(line,"photo") != NULL) su->kind_of_att = PHOTO_ATT;
	if (strstr(line,"ideal") != NULL) su->kind_of_att = IDEAL;
      } else if (strstr(line,"projgen") != NULL) { //maja 2005-12 
	if (strstr(line,"kohler") != NULL)   su->kind_of_projgen = KOHLER;
	if (strstr(line,"siddon") != NULL)   su->kind_of_projgen = SIDDON;
	if (strstr(line,"joseph") != NULL)   su->kind_of_projgen = JOSEPH;
	if (strstr(line,"josephnn") != NULL) su->kind_of_projgen = JOSEPHNN;
      } else if (strstr(line,"voxelization") != NULL) {
	sscanf(lineptr,"%s",su->vox_file);
	su->vox = 1;
      } else if (strstr(line, "voxelpoints") != NULL) {
	sscanf(lineptr,"%i", &su->voxpt);
      } else if (strstr(line, "voxelsx") != NULL) {
	sscanf(lineptr,"%i", &su->voxels[0]);
      } else if (strstr(line, "voxelsy") != NULL) {
	sscanf(lineptr,"%i", &su->voxels[1]);
      } else if (strstr(line, "voxelsz") != NULL) {
	sscanf(lineptr,"%i", &su->voxels[2]);
      } else if (strstr(line, "voxlenx") != NULL) {
	sscanf(lineptr,"%lf", &su->voxlen[0]);
      } else if (strstr(line, "voxleny") != NULL) {
	sscanf(lineptr,"%lf", &su->voxlen[1]);
      } else if (strstr(line, "voxlenz") != NULL) {
	sscanf(lineptr,"%lf", &su->voxlen[2]);
      } else if (strstr(line, "voxelcenter") != NULL) {
	sscanf(lineptr,"%lf %lf %lf", &su->vox_center[0], 
	       &su->vox_center[1], &su->vox_center[2]);
      } else if (strstr(line, "voxelsize") != NULL) {
	sscanf(lineptr,"%lf %lf %lf", &su->voxlen[0], 
	       &su->voxlen[1], &su->voxlen[2]);
      } else if (strstr(line, "voxelnr") != NULL) {
	sscanf(lineptr,"%i %i %i", &su->voxels[0], 
	       &su->voxels[1], &su->voxels[2]);
      } else if (strstr(line,"energyspectrum") !=NULL) {
	sscanf(lineptr,"%s",su->spec_file);
      } else if (strstr(line,"mono") !=NULL) {
	sscanf(lineptr, "%lf", &su->mono);
      } else if (strstr(line,"response") !=NULL) {
	sscanf(lineptr,"%s",su->response_file);
      } else if (strstr(line,"nrofphotons") !=NULL) {
	sscanf(lineptr,"%lf", &su->nr_ph);
	su->noise = 1;
      } else {
	printf("Unknown parameter %s in %s\n", line, su->ini_file);
      }
    } 
  }
  fclose(fp);

#ifdef STANDALONE
  sprintf(su->type,"STANDALONE");
  /*---> check if directories exist <---*/
  if (su->vox==0) {  /* if projection is not switched off */
    fp=fopen(su->pro_file,"wb");
    if (fp==NULL) {
      printf("Unable to create projection file %s.",
	     su->pro_file);
      printf("\nMake sure that directory exists.\n");
      hejda("Terminated.\n",9);
    } else 
      fclose(fp);
  }

  if (su->vox==1) {  /* if voxelization */
    fp=fopen(su->vox_file,"wb");
    if (fp==NULL) {
      printf("Unable to create voxel file\n\"%s\".",
	     su->vox_file);
      printf("\nMake sure that directory exists.\n");
      hejda("Terminated.\n",9);
    } else 
      fclose(fp);
  }
#else
  sprintf(su->type,"MATLAB");
  read_command_init(pt, argc-2, argv+2);
#endif
}

/****************************************************************************
 * reads init commands from MATLAB                                          *
 *                                                                          *
 * author: Olle Seger                                             2003-2005 *
 *                                                                          *
 * Input: setup parameters                                                  *
 * Return: ./. (void)                                                       *
 ****************************************************************************/
void read_command_init(TAKEPAR *pt, int argc, char **argv)
{
  int i;
  char *lineptr, *line;
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;

  if (argc>0) {
    for (i=0; i<argc; i++) {
      line = argv[i];
      lineptr = strchr(line,'=');
      lineptr++;
      if (strstr(line,"verb") != NULL) {
	su->verbose=1;
      } else if (strstr(line,"nonoise") != NULL) {
	su->noise=0;
      } else if (strstr(line,"debug") != NULL) {
	su->debug=1;
      } else if (strstr(line,"phantom") != NULL) {
	sscanf(lineptr,"%s",su->ph_file);
      }else if (strstr(line,"trajectory") != NULL) {
	sscanf(lineptr,"%s",su->trj_file);
      } else if (strstr(line,"detector") != NULL) {
	sscanf(lineptr,"%s", su->det_file);
      } else if (strstr(line,"attenuation") != NULL) {
	if (strstr(line,"log") != NULL)   su->kind_of_att = TAKELOG;
	if (strstr(line,"nolog") != NULL) su->kind_of_att = NOLOG;
	if (strstr(line,"photo") != NULL) su->kind_of_att = PHOTO_ATT;
	if (strstr(line,"ideal") != NULL) su->kind_of_att = IDEAL;
      } else if (strstr(line,"projgen") != NULL) { //maja 2005-12
	if (strstr(line,"kohler") != NULL)   su->kind_of_projgen = KOHLER;
	if (strstr(line,"siddon") != NULL)   su->kind_of_projgen = SIDDON;
	if (strstr(line,"joseph") != NULL)   su->kind_of_projgen = JOSEPH;
	if (strstr(line,"josephnn") != NULL) su->kind_of_projgen = JOSEPHNN;
      } else if (strstr(line,"voxeliz") != NULL) {
	su->vox = 1;
      } else if (strstr(line, "voxelcenter") != NULL) {
	sscanf(lineptr,"%lf %lf %lf", &su->vox_center[0], 
	       &su->vox_center[1], &su->vox_center[2]);
      } else if (strstr(line, "voxelsize") != NULL) {
	sscanf(lineptr,"%lf %lf %lf", &su->voxlen[0], 
	       &su->voxlen[1], &su->voxlen[2]);
      } else if (strstr(line, "voxelnr") != NULL) {
	sscanf(lineptr,"%lf %lf %lf", &su->voxels[0], 
	       &su->voxels[1], &su->voxels[2]);
      } else if (strstr(line, "initrot") != NULL) {
	sscanf(lineptr,"%lf", &pd->initrot);
      } else if (strstr(line,"energyspectrum") !=NULL) {
	sscanf(lineptr, "%s", su->spec_file);
      } else if (strstr(line,"mono") !=NULL) {
	sscanf(lineptr, "%lf", &su->mono);
      } else if (strstr(line,"noise") !=NULL) {
	su->noise = 1;
      }
    }
  }
}

/****************************************************************************
 * prints input data to log file                                            *
 *                                                                          *
 * author: Olle Seger                                             2003-2005 *
 ****************************************************************************/
void print_log(TAKEPAR *pt)
{
  FILE *log;            /* detector file */
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;
  OBJECT *po = su->po;
  POLY *pp = su->pp;
  TRAJECT *ptrj = pd->to_trj;
  int i,x,y, E;

  log = fopen(su->log_file, "w");
  if (log == NULL)
    hejda(su->log_file, 6);

  fprintf(log, "\n***** Setup parameters: *****\n");
  fprintf(log, "Type: %s\n", su->type);
  fprintf(log, "Init file: %s\n", su->ini_file);
  fprintf(log, "Phantom file: %s\n", su->ph_file);
  fprintf(log, "Detector file: %s\n", su->det_file);
  fprintf(log, "Projection file: %s\n", su->pro_file);
  fprintf(log, "Log file: %s\n", su->log_file);
  fprintf(log, "Trajectory file: %s\n", su->trj_file);
  fprintf(log, "Projection file: %s\n", su->pro_file);

  fprintf(log, "Scatter=%d\n", su->scatter);
  fprintf(log, "Scatter file: %s\n", su->scatter_file);

  fprintf(log, "Spectrum file: %s\n", su->spec_file);
  fprintf(log, "Mono energy = %5.1f\n", su->mono);
  fprintf(log, "Mono index = %d\n", su->mono_i);
  fprintf(log, "Detector response file: %s\n", su->response_file);
  fprintf(log, "Nr of E levels=%d\n", su->pp->lvls);
  fprintf(log, "Nr of materials: %d\n", su->nr_of_mat);
  for (i=0; i<su->nr_of_mat; i++) 
    if (su->mat_flag[i])
      fprintf(log, "  %d: %s\n", i, su->mu_file[i]);
   
  fprintf(log, "Filter=%d\n", su->filter);
  fprintf(log, "Noise=%d\n", su->noise);
  fprintf(log, "Nr of photons=%f\n", su->nr_ph);

  fprintf(log, "Voxelize=%d\n", su->vox);
  fprintf(log, "Nr of voxels: %d %d %d\n", 
	  su->voxels[0],su->voxels[1],su->voxels[2] );
  fprintf(log, "Volume size: %f %f %f\n", 
	  su->voxlen[0],su->voxlen[1],su->voxlen[2] );
  fprintf(log, "Volume center: %f %f %f\n", 
	  su->vox_center[0],su->vox_center[1],su->vox_center[2] );

  if (su->kind_of_att == TAKELOG)
    fprintf(log, "Attenuation=TAKELOG\n");
  else if (su->kind_of_att == PHOTO_ATT)
    fprintf(log, "Attenuation=PHOTO_ATT\n");
  else if (su->kind_of_att == IDEAL)
    fprintf(log, "Attenuation=IDEAL\n");
  else
    fprintf(log, "Attenuation=NOLOG\n");
    
  if (su->kind_of_projgen == KOHLER) //maja 2005-12
    fprintf(log, "Projection generation=KOHLER\n");
  else if (su->kind_of_projgen == SIDDON)
    fprintf(log, "Projection generation=SIDDON\n");
  else if (su->kind_of_projgen == JOSEPH)
    fprintf(log, "Projection generation=JOSEPH\n");
  else
    fprintf(log, "Projection generation=JOSEPHNN\n");
    
  fprintf(log, "Verbose=%d\n", su->verbose);
  fprintf(log, "Debug=%d\n", su->debug);
  fprintf(log, "Nr of objects=%d\n", su->nr_obj);

  fprintf(log, "\n***** Detector parameters: *****\n");
  fprintf(log, "Length of detector: %f %f\n", pd->length_x, pd->length_y);
  fprintf(log, "Size of detector: %d %d\n", pd->pixel_x, pd->pixel_y);
  fprintf(log, "Nr of projections: %d\n", pd->nr_of_prj);
  fprintf(log, "Points_xy: %d\n", pd->points_xy);
  fprintf(log, "Source width: %f %f\n", pd->source_width, pd->source_height);
  fprintf(log, "Position of first sub sample on source: %f %f\n", 
	  pd->src_x_0, pd->src_y_0);
  fprintf(log, "Distance between subsamples on source: %f %f \n", 
	  pd->src_x_delta, pd->src_y_delta);
  fprintf(log, "Length of pixel in x-, y-direction: %f %f\n", 
	  pd->pix_lgth_x, pd->pix_lgth_y);
  fprintf(log, "Fanangle: %f\n", pd->fanangle);
  fprintf(log, "Skew: %f\n", pd->skew);
  fprintf(log, "Channel offset: %f\n", pd->channel_offset);
  fprintf(log, "Cylinder radius: %f\n", pd->cylinder_radius);
  fprintf(log, "Shape: %d\n", pd->shape);
  fprintf(log, "Mirror points: %d\n", pd->mi_nr);
  fprintf(log, "Nr of rays: %d\n", pd->nr_rays);

  fprintf(log, "\n***** Objects in the phantom: *****\n");
  fprintf(log, "Nr: kind mat dens  phantom ratio\n");
  for (i=0; i<su->nr_obj; i++, po=po->next) {
    fprintf(log, "%d:   %d    %d  %4.2f %d %4.2f\n", 
	    i, po->kind_of_obj, po->mat, po->dens, po->phm, 100.0*
	    po->nrhits/(pd->nr_of_prj*pd->pixel_x*pd->pixel_y*pd->nr_rays));
    for(y=0; y<3; y++) {
      fprintf(log, "  ");
      for(x=0; x<4; x++) {
        fprintf(log, "%4.3f ", po->obj_pos[y][x]);
      }
      fprintf(log, "\n");
    }
    fprintf(log, "\n");
  }

  if (ptrj!=NULL) {		/* Trajectory data? */
    fprintf(log, "\n***** Trajectory parameters: *****\n");
    if (pd->initrot) {
      fprintf(log, "initial rotation: fi=%5.2f\n", pd->initrot);
      for (i=0; i<3; i++) {
	fprintf(log, "%5.2f %5.2f %5.2f %5.2f\n",
		pd->R[i][0],
		pd->R[i][1],
		pd->R[i][2],
		pd->R[i][3]);
      }
    }
    fprintf(log, "\nsrc start: x=%5.2f y=%5.2f z=%5.2f\n", 
	    ptrj->srcpos[0],ptrj->srcpos[1],ptrj->srcpos[2]);
    fprintf(log, "first detector pos:\n");
    for (i=0; i<3; i++) {
      fprintf(log, "%5.2f %5.2f %5.2f %5.2f\n",
	      ptrj->detpos[i][0],
	      ptrj->detpos[i][1],
	      ptrj->detpos[i][2],
	      ptrj->detpos[i][3]);
    }
    ptrj += (pd->nr_of_prj - 1); 
    fprintf(log, "\nsrc end: x=%5.2f y=%5.2f z=%5.2f\n", 
	    ptrj->srcpos[0],ptrj->srcpos[1],ptrj->srcpos[2]);
    fprintf(log, "last detector pos:\n");
    for (i=0; i<3; i++) {
      fprintf(log, "%5.2f %5.2f %5.2f %5.2f\n",
	      ptrj->detpos[i][0],
	      ptrj->detpos[i][1],
	      ptrj->detpos[i][2],
	      ptrj->detpos[i][3]);
    }
  }

  if (su->mat & su->spect) {
    fprintf(log, "\n***** Polychromatic parameters: *****\n");
    fprintf(log, "Normalized Spectrum\n");
    fprintf(log, "k = Nr_of_rays * Nr_of_photons\n");
    fprintf(log, "SUM = sum[N(E) * dE]\n");
    fprintf(log, " Nr:  lambda(kEV)  k*N(E)*dE/SUM  k*E*N(E)*dE/SUM");
    for (i=0; i<su->nr_of_mat; i++)
      if (su->mat_flag[i])
	fprintf(log, "  mu_%d/rho", i);
    fprintf(log, "\n");
    
    for (E=0; E < pp->lvls; E++) {
      fprintf(log, "%3d: %12.1f %14.4f %16.3f ", 
	      E, pp->energy[E], pp->nrp[E], 
	      pp->energy[E]*pp->nrp[E] );
      for (i=0; i<su->nr_of_mat; i++)
	if (su->mat_flag[i])
	  fprintf(log, " %8.1f", pp->mu[E + i*pp->lvls]);
      fprintf(log, "\n");
    }
  }
  
  fclose(log);
}

/****************************************************************************
 * reads the detector specification parameters from file <det_file>.        *
 *                                                                          *
 * author: Jens Mueller-Merbach                                      dec 95 *
 *                                                                          *
 * Input: setup parameters                                                  *
 * Output: detector parameters                                              *
 * Return: ./. (void)                                                       *
 ****************************************************************************/
void read_detector(TAKEPAR *pt)
{
  FILE *detector;            /* detector file */
  char *lineptr;             /* linepointer */
  char line[STR_LEN];        /* the line */
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;

  if (su->vox)
    return;

  if (su->debug)
    printf("\nLoading detector data from file %s\n",
	   su->det_file); 

  detector = fopen(su->det_file, "rt");
  if (detector == NULL) hejda(su->det_file,4);

  /*----------------*
   * Default values *
   *----------------*/
  pd->shape = SHAPE_PLANAR;
  pd->source_width = 0.0;
  pd->source_height = 0.0;
  pd->channel_offset = 0.0;
  pd->fanangle = 0.0;
  pd->skew = 0.0;
  pd->length_x = 0.0;
  pd->length_y = 1.0;
  pd->points_xy=1;
  pd->pixel_y=1;
  pd->pixel_x=1;
  pd->to_trj = NULL;

  while(my_fgets(line, detector) != NULL) {
    lineptr = line;
    if ((lineptr = strchr(line,'=')) != NULL) {
      lineptr++;               /* position the lineptr after the =-sign */
      if (strstr(line,"xlen") != NULL) {
	sscanf(lineptr,"%lf", &pd->length_x);
      } else if (strstr(line,"ylen") != NULL) {
	sscanf(lineptr,"%lf", &pd->length_y);
      } else if (strstr(line,"sourceWidth") != NULL) {
	sscanf(lineptr,"%lf", &pd->source_width);
      } else if (strstr(line,"sourceHeight") != NULL) {
	sscanf(lineptr,"%lf", &pd->source_height);
      } else if (strstr(line,"xpix") != NULL) {
	sscanf(lineptr,"%d", &pd->pixel_x);
      } else if (strstr(line,"ypix") != NULL) {
	sscanf(lineptr,"%d", &pd->pixel_y);
      } else if (strstr(line,"xypoints") != NULL) {
	sscanf(lineptr,"%d", &pd->points_xy);
      } else if (strstr(line,"shape") != NULL) {
	char shape_str[255];
 
	sscanf(lineptr, "%s", shape_str);
	if ( strcmp( shape_str, "cylindricalAroundSource" ) == 0) {
	  pd->shape = SHAPE_CYLINDRICAL_AROUND_SOURCE;
	} else if ( strcmp( shape_str, "cylindricalAroundRotationAxis" ) == 0) {
	  pd->shape = SHAPE_CYLINDRICAL_AROUND_ROTATION_AXIS;
	} else if ( strcmp( shape_str, "planar" ) == 0 ) {
	  pd->shape = SHAPE_PLANAR;
	} else {
	  hejda(shape_str,7);
	}
      } else if (strstr(line,"fanangle") != NULL) {
	sscanf(lineptr,"%lf", &pd->fanangle);
      } else if (strstr(line,"height") != NULL) {
	sscanf(lineptr,"%lf", &pd->length_y);
      } else if (strstr(line,"skew") != NULL) {
	sscanf(lineptr,"%lf",&pd->skew);
      } else if (strstr(line,"channel_offset") != NULL ) {
	sscanf(lineptr,"%lf", &pd->channel_offset);
      } else if (strstr(line,"channels") != NULL) {
	sscanf(lineptr,"%d", &pd->pixel_x);
      } else if (strstr(line,"rows") != NULL) {
	sscanf(lineptr,"%d", &pd->pixel_y);
      } else 
	hejda(line,7);
    } 
  }
  
  pd->src_x_delta = pd->source_width / (double)pd->points_xy;
  pd->src_y_delta = pd->source_height / (double)pd->points_xy;

  pd->src_x_0 = - 0.5 * (pd->source_width - pd->src_x_delta);
  pd->src_y_0 = - 0.5 * (pd->source_height - pd->src_y_delta);
        
  pd->pix_lgth_x = pd->length_x / (double)pd->pixel_x;
  pd->pix_lgth_y = pd->length_y / (double)pd->pixel_y;

  if ((pd->points_xy > 1) && ((pd->source_width > 0.0) || 
			      (pd->source_height > 0.0))) {
    pd->mi_nr = 2;
  } else  {
    pd->mi_nr = 1;  /* only one sub point or point source => no mirroring */
  }
  pd->nr_rays = pd->mi_nr*pd->points_xy*pd->points_xy;
  fclose(detector);
} 

/****************************************************************************
 * reads the input file phantom.file that contains all phantom creating     *
 * data (shape, lengths/radii, translation, rotation, density). From that   *
 * it generates the transformation matrices obj_pos for all objects.        *
 *                                                                          *
 * author: Jens Mueller-Merbach                                      dec 95 *
 *                                                                v2 jan 96 *
 *                                                                          *
 * Input: name of phantom file                                              *
 * Output: number of objects, kind of objects                               *
 * Return: objects (pointer)                                                *
 ****************************************************************************/
OBJECT *read_phantom (TAKEPAR *pt) 
{
  OBJECT *po;
  FILE *fp;			/* pointer to phantom.file */
  char str_s[STR_LEN], str2_s[STR_LEN];
  char *str, *str2;
  char line_s[STR_LEN], *line;	/* one line of the input file */
  double a,b,c,x,y,z;	        /* radii/lengths of object, translation */
  double ang1,ang2,ang3;	/* rotation angles */
  char ax1, ax2, ax3;		/* rotation axes */
  double den;			/* density */
  int mat=0;			/* material */
  int i;			/* counter */
  char order;
  int nr_obj;
  int phm = 1;
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;

  if (su->debug)
    printf("\nLoading phantom data from\nfile \"%s\".\n", su->ph_file); 

  fp = fopen(su->ph_file, "rt");
     
  if (fp == NULL) 
    hejda(su->ph_file,4);

  /* Default values */
  order = 0;
  nr_obj = 0;
  ang1=ang2=ang3=0.0;

  line = line_s;
  str = str_s; 
  str2 = str2_s;

  /*----------------------*
   * decoding information *
   *----------------------*/
  while (my_fgets(line_s, fp) != NULL) {
    if (strstr(line_s, "[filter]")) {
      phm=0;
      su->filter = 1;
    } else if (strstr(line_s, "material")) {
      if ((line = strchr(line_s,'=')) != NULL) {
	line++;
	sscanf(line, "%d%s", &mat, str_s);
	sprintf(su->mu_file[mat], "%s", str_s);
      }
    } else if ((line[0] == 'e')||(line[0] == 'c')
	       ||(line[0] == 'b')||(line[0] == 'v')) {

      /* we have an object ! */
      nr_obj++;

      po = (OBJECT *) CALLOC (1, sizeof(OBJECT));

      if (po == NULL) 
	hejda("su->po",1);

      /* insert the empty object first in the list */
      po->next = su->po;
      su->po = po;
      po->phm = phm;
      po->nrhits = 0;

      /*-----------------------------------------*
       * ellipsoid, cylinder, box or voxel data? *
       *-----------------------------------------*/
      switch (line[0]){
      case 'e':
	po->kind_of_obj = OBJECT_ELLIPSOID;         
	break;
      case 'c':
	po->kind_of_obj = OBJECT_CYLINDER;      
	break;
      case 'b':
	po->kind_of_obj = OBJECT_BOX;       
	break;
      case 'v':
	po->kind_of_obj = OBJECT_VOXELS;        
	break;
      }

      /*-----------------------------*
       * read radii and translations *
       *-----------------------------*/
      sscanf(line,
	     "%*s a=%lf b=%lf c=%lf x=%lf y=%lf z=%lf",&a,&b,&c,&x,&y,&z);

      /*------------------------------------*
       * what comes next? move line pointer *
       *------------------------------------*/
      line=strstr(line,"z=");

      if (line == NULL) hejda("fp file (z)",7);
      line +=2;
      line=strchr(line,' ');

      ax1 = 0;
      ax2 = 0;
      ax3 = 0;

      /*----------------------------------------*
       * read rotational parameters and density *
       *----------------------------------------*/
      sscanf(line,"%s",str);

      if (strstr(str, "dens") == NULL) {
	if (strstr(str, "theta") != NULL) {

	  /* read theta */
	  str2=strchr(str,'=')+1;
	  ax1 = 3;
	  sscanf(str2,"%lf",&ang1);

	  /* read phi */
	  str=strstr(line, "phi");
	  sscanf(str,"%s",str2);

	  if(strstr(str2,"phi") == NULL) 
	    hejda("phantom file (phi)",7);
	  str=strchr(str2,'=')+1;
	  ax2 = 2;
	  sscanf(str,"%lf",&ang2);
            
	  if (po->kind_of_obj != OBJECT_VOXELS) {
	    /* read density */
	    str=strstr(line,"dens");
	    sscanf(str,"%s",str2);

	    if (strstr(str2, "dens") == NULL) 
	      hejda("phantom file (dens)",7);
	    str=strchr(str2,'=')+1;

	    sscanf(str,"%lf",&den);
	  } else {
	    char filename[999];

	    /* read voxeldata */
	    str=strstr(line,"file");
	    sscanf(str,"%s",str2);

	    if (strstr(str2, "file") == NULL) 
	      hejda("phantom file (file)",7);
	    str=strchr(str2,'=')+1;

	    sscanf(str,"%s", filename);
	    po->voxeldata = 
	      read_volume_with_margin(su, filename, po->Nvox);
	  }

	  /* read material if not ideal */
	  if (su->kind_of_att != 3) {
	    str = strstr(line,"mat");
	    if (str != NULL) {
	      str2 = strchr(str,'=')+1;
	      sscanf(str2,"%i", &mat);
	    }
	  }
	} else if (strstr(str, "rot") != NULL) {
	  char *du[5];
	  for(i=0; i<=4; du[i++]=(char *)MALLOC(12*sizeof(char)));
	  sscanf(line,"%s %s %s %s %s",du[0],du[1],du[2],du[3],du[4]);

	  if(strstr(du[0],"rot=")!=NULL) {

	    /* rot. about local axes */
	    for (i=0;i<=4;i++) {
	      if(strstr(du[i],"dens")!=NULL) {
		du[i]=strchr(du[i],'=')+1;

		sscanf(du[i],"%lf",&den);
		i=3;
	      } else if(strstr(du[i],"xrot")!=NULL) {
		double angle;
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&angle);
		(i==0)?(ax1=1,ang1=angle):
		  ((i==1)?(ax2=1,ang2=angle):
		   (ax3=1,ang3=angle));
	      } else if(strstr(du[i],"yrot")!=NULL) {
		double angle;
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&angle);
		(i==0)?(ax1=2,ang1=angle):
		  ((i==1)?(ax2=2,ang2=angle):
		   (ax3=2,ang3=angle));
	      } else if(strstr(du[i],"zrot")!=NULL) {
		double angle;
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&angle);
		(i==0)?(ax1=3,ang1=angle):
		  ((i==1)?(ax2=3,ang2=angle):
		   (ax3=3,ang3=angle));
	      } else if (strstr(du[i],"mat")!=NULL) {
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%i",&mat);
	      }
	    }
	  } else  { /* rotate about reference axes */
	    order=1;
	    for (i=0;i<=4;i++) {
	      if(strstr(du[i],"dens")!=NULL) {
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&den);
		i=3;
	      } else if(strstr(du[i],"rotx")!=NULL) {
		double angle;
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&angle);
		(i==0)?(ax3=1,ang3=angle):
		  ((i==1)?(ax2=1,ang2=angle):
		   (ax1=1,ang1=angle));
	      } else if(strstr(du[i],"roty")!=NULL) {
		double angle;
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&angle);
		(i==0)?(ax3=2,ang3=angle):
		  ((i==1)?(ax2=2,ang2=angle):
		   (ax1=2,ang1=angle));
	      } else if(strstr(du[i],"rotz")!=NULL) {
		double angle;

		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%lf",&angle);
		(i==0)?(ax3=3,ang3=angle):
		  ((i==1)?(ax2=3,ang2=angle):
		   (ax3=1,ang1=angle));
	      } else if (strstr(du[i],"mat")!=NULL) {
		du[i]=strchr(du[i],'=')+1;
		sscanf(du[i],"%i",&mat);
	      }
	    }
	  }
          
	  while (ax1 == 0) {
	    ax1 = ax2; ax2 = ax3; ax3 = 0;
	    ang1=ang2; ang2=ang3;
	  }
	  /*for(i=0;i<=3; free(du[i]),i++);*/
	}

      } else { /* read density if no rotation defined */
	line += (strlen(str)+1);
	if (strstr(str, "dens") == NULL) 
	  hejda("phantom file (dens)",7);
	str=strchr(str,'=')+1;
	sscanf(str,"%lf",&den);

	/* read material if not ideal */
	if (su->kind_of_att != 3) {
	  str=strstr(line,"mat");
	  str2=strchr(str,'=')+1;
	  sscanf(str2,"%i",&mat);
	}
      }
      po->dens = den;
      po->mat = mat;

      generate_mrx(po, a, b, c, x, y, z, ang1, ang2, ang3, ax1, ax2, ax3, order);
    }
    line = line_s;
    str = str_s;
    str2 = str2_s;
  }

  fclose(fp);

  if (su->debug)
    printf("Nr of objects: %i\n", nr_obj); 

  su->nr_obj = nr_obj;
  return (su->po);
}

/****************************************************************************
 * reads the trajectory file.                                               *
 *                                                                          *
 * author: Jens Mueller-Merbach                                      feb 96 *
 *                                                                          *
 * Input: name of trj-file                                                  *
 * Output: detector and source positions                                    *
 * Return:                                                                  *
 ****************************************************************************/
TRAJECT *read_trj(TAKEPAR *pt)
{
  FILE *fp;
  char *lineptr, line[STR_LEN];
  TRAJECT *ptrj = NULL;
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;

  if (su->vox == 1)
    return(ptrj);

  if (su->debug)
    printf("\nLoading trajectory data from\nfile \"%s\".\n",
	   su->trj_file);
  if ((fp = fopen(su->trj_file, "rt"))==NULL) hejda(su->trj_file,4);

  pd->nr_of_prj = 1;		/* Default value */
  
  while((lineptr=my_fgets(line, fp)) != NULL) {
    if (strstr(line,"projections") != NULL) {
      lineptr = strchr(line,'=');
      sscanf(++lineptr,"%d", &pd->nr_of_prj);
      ptrj=pd->to_trj=(TRAJECT *) MALLOC(pd->nr_of_prj * sizeof(TRAJECT));
      if (ptrj==NULL) 
      hejda("TRAJECT *ptrj", 1);
    } else if (strstr(line,"explicit") != NULL) {
      int i;
      for(i=0; i < pd->nr_of_prj; i++) {
	int j;
	my_fgets(line, fp);

	/* read source position i (3 datas) */
	sscanf(line,"%lf %lf %lf",
	       &pd->srcpos[0],
	       &pd->srcpos[1],
	       &pd->srcpos[2]);
           
	/* read detector position i (12 datas) */
	for(j=0; j<3; j++) {
	  my_fgets(line, fp);
	  sscanf(line,"%lf %lf %lf %lf",
		 &pd->detpos[j][0],
		 &pd->detpos[j][1],
		 &pd->detpos[j][2],
		 &pd->detpos[j][3]);
	}
      }
    } else {
      int j;
      MATRIX tfmm;  /* TransForMation Matrix */
      if (strstr(line,"implicit") != NULL) 
        my_fgets(line, fp);
        
      /* read first source position (3 datas) */
      sscanf(line,"%lf %lf %lf",
	     &pd->srcpos[0],
	     &pd->srcpos[1],
	     &pd->srcpos[2]);

      /* read first detector position (12 datas) */
      for(j=0; j<3; j++) {
      my_fgets(line, fp);
      sscanf(line,"%lf %lf %lf %lf",
	       &pd->detpos[j][0],
	       &pd->detpos[j][1],
	       &pd->detpos[j][2],
	       &pd->detpos[j][3]);
      }

      /* read transformation matrix (12 datas) */
      for(j=0; j<3; j++) {
	my_fgets(line, fp);
	sscanf(line,"%lf %lf %lf %lf",
	       &tfmm[j][0],
	       &tfmm[j][1],
	       &tfmm[j][2],
	       &tfmm[j][3]);
      }

      /* do an initial rotation */
      if (fabs(pd->initrot)>0.01) 
	rot_matrix(3, pd->initrot, pd->R);
      else 
	rot_matrix(3, 0.0, pd->R);
      
      trfm(pd->srcpos, pd->R, ptrj->srcpos);
      mult_matrix(pd->detpos, pd->R, ptrj->detpos);

      /* compute position j recursively from position j-1 */
      for(j=1; j < pd->nr_of_prj; j++, ptrj++) {
	trfm(ptrj->srcpos, tfmm, (ptrj+1)->srcpos);
	mult_matrix(tfmm, ptrj->detpos, (ptrj+1)->detpos);
      }

    }
  }

  pd->cylinder_radius = sqrt(pd->to_trj->srcpos[0] * pd->to_trj->srcpos[0] +
			     pd->to_trj->srcpos[1] * pd->to_trj->srcpos[1]);
  pd->source2det = sqrt((pd->to_trj->srcpos[0] - pd->detpos[0][3]) *
			(pd->to_trj->srcpos[0] - pd->detpos[0][3]) +
			(pd->to_trj->srcpos[1] - pd->detpos[1][3]) *
			(pd->to_trj->srcpos[1] - pd->detpos[1][3]) ); //maja 2005-11

  fclose(fp);
  return (pd->to_trj);
}

/****************************************************************************
 * Reads voxeldata from file.                                               *
 * Returned data includes a margin around the volume data.                  *
 *                                                                          *
 * author: Henrik Turbell                                            jun 99 *
 *                                                                          *
 * Input: filename                                                          *
 * Output: number of voxels in each dimension                               *
 * Return: pointer to voxel data                                            *
 ****************************************************************************/
float* read_volume_with_margin(SETUP *su, char *filename, int Nvox[3])
{
  FILE *infile;
  float *voxelValue;
  int y,z;
  float* pos;
  int Nx, Ny, Nz, Nxy, Nxyz;
  int Mx, My, Mz, Mxy, Mxyz; 

  if ( NULL == ( infile = fopen( filename, "rb" ) ) )
    {
      printf("Unable to open file '%s'.\n", filename);
      hejda("Terminated",9);;
    }

  fread( &Nvox[0], sizeof( int ), 1, infile);
  fread( &Nvox[1], sizeof( int ), 1, infile);
  fread( &Nvox[2], sizeof( int ), 1, infile);
  
  Nx = Nvox[0];
  Ny = Nvox[1];
  Nz = Nvox[2];
  Nxy = Nx * Ny;
  Nxyz = Nxy * Nz;

  Mx = Nx + 2 * VOXMARG;
  My = Ny + 2 * VOXMARG;
  Mz = Nz + 2 * VOXMARG;
  Mxy = Mx * My;
  Mxyz = Mxy * Mz;

  if (su->debug)
    printf( "Reading voxel file '%s' ( %d * %d * %d voxels )",
	    filename, Nx, Ny, Nz );
   
  if ( NULL == ( voxelValue = (float *) CALLOC(Mxyz, sizeof(float)))) {
    printf("voxvis: unable to allocate voxel values.\n");
    hejda("Terminated",9);;
  }
   
  for (z = VOXMARG; z < Nz+VOXMARG; z++) {
    pos = voxelValue + z * Mxy + VOXMARG;
    for (y = VOXMARG; y < Ny+VOXMARG; y++) {       
      if ( Nx != fread( pos + y * Mx,
			sizeof( float ),
			Nx,
			infile ) )      {
	printf("voxvis: unable to read all voxel values.\n");
	hejda("Terminated",9);;
      }       
    }
  }
 
  fclose( infile );

  return voxelValue;
}

/****************************************************************************
 * terminates the program after giving some information about the reason    *
 * on stderr.                                                               *
 *                                                                          *
 * author: Jens Mueller-Merbach                                      nov 95 *
 *                                                                          *
 * Input: name of reason, kind of reason                                    *
 * Return: ./. (exit)                                                       *
 ****************************************************************************/
void hejda (char *name, int reason)
{
  char error_msg[80];

  switch (reason)
    {
    case 1:
      sprintf(error_msg, "\n*** Error when allocating memory for pointer %s.\n",
	      name);
      break;
    case 2:
      sprintf(error_msg, "\n*** Error: Singular matrix in function %s.\n",name);
      break;
    case 3:
      sprintf(error_msg, "\n*** Error: Cannot rotate about axis %s.\n",name);
      break;
    case 4:
      sprintf(error_msg, "\n*** Error: %s not found.\n", name);
      break;
    case 5:
      sprintf(error_msg, "\n*** Error: %s had illegal value.\n", name);
      break;
    case 6:
      sprintf(error_msg, "\n*** Error: Could not open %s.\n", name);
      break;
    case 7:
      sprintf(error_msg, "\n*** Error: Could not interprete %s.\n", name);
      break;
    case 8:
      sprintf(error_msg, "\n*** Error in %s: Unexpected value.\n", name);
      break;
    case 9:
      sprintf(error_msg, "%s\n", name);
      break;
    case 10:
      sprintf(error_msg, "\n*** Error: Spectrum file is missing. %s\n", name);
      break;
    case 11:
      sprintf(error_msg, "\n*** Error: Material file is missing. %s\n", name);
      break;
    case 12:
      sprintf(error_msg, 
	      "\n*** Error: Wanted mono energy not included in spectrum file. %s\n", name);
      break;
    }

#ifdef STANDALONE
  printf("%s", error_msg);
  exit(errno);
#else
  mexErrMsgTxt(error_msg);
#endif
}
