/*******************************************************************
 * Calculation of intersection with ellipses, cylinders, boxes and *
 * voxel volumes.                                                  *
 *                                                                 *
 * Written by Jens Mueller-Merbach, 1996                           *
 * Henrik Turbell, 1999-2000                                       *
 * Maria Magnusson Seger, 2005                                     *
 *******************************************************************/

/*
The program take is licenced under the X11 license:

Copyright (C), 1996-2005, 
Jens Mueller-Merbach, Maria Magnusson, Henrik Turbell, Olle Seger

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>        /* for printf(), ... */
#include <stdlib.h>
#include <errno.h>        /* for errno */
#include <malloc.h>       /* for malloc(), realloc(), free() */
#include <string.h>       /* for strstr(), strcpy(), ... */
#include <math.h>         /* for sqrt(), cos(), log(), ... */
#include "take.h"

/************************************
 * Transforms two points p1 and p2  *
 * and calculates the distance.     *
 ************************************/
double length (POINT p1, POINT p2, OBJECT *po)
{
  double len2=0.0, len;		/* length */
  int i;			/* counter */
  POINT x1, x2;

  trfm(p1, po->obj_pos, x1);	/* retransform the points */
  trfm(p2, po->obj_pos, x2);

  for (i=0; i<3; i++) {
    len = x1[i]-x2[i];
    len2 += len*len;
  }
  return (sqrt(len2));
}

/***************************************************
 * Calculate a point on the detector.              *
 * maja 2005-11: several (double) casting inserted *
 ***************************************************/
void calc_detector(DETECTOR *pd, int xpix, int ypix, int xpoint, int ypoint, 
		   POINT ray_detector)
{
  double gamma;
  
  ray_detector[1] = 0.0;
  
  switch ( pd->shape ) { 		
  case SHAPE_PLANAR:
    ray_detector[0] = - 0.5 * pd->length_x +
      ((double)xpix + (double)pd->channel_offset + 
       ((double)xpoint + 0.5) / (double)pd->points_xy) * pd->pix_lgth_x;
    ray_detector[2] = - 0.5 * pd->length_y + 
      ((double)ypix + ((double)ypoint + 0.5) 
       / (double)pd->points_xy) * pd->pix_lgth_y;
    break;
  case SHAPE_CYLINDRICAL_AROUND_SOURCE:
    gamma = ( ( (double)xpix + (double)pd->channel_offset + 
		( (double)xpoint + 0.5 ) / (double)pd->points_xy )
	      / (double)pd->pixel_x * 2.0 - 1.0 ) * pd->fanangle;

    ray_detector[0] = pd->source2det * tan(gamma); //maja 2005-11

    ray_detector[2] = ( -0.5 * pd->length_y +
			( (double)ypix + ((double)ypoint + 0.5) 
			  / (double)pd->points_xy ) 
			* pd->pix_lgth_y ) /  cos(gamma);
    break;
  case SHAPE_CYLINDRICAL_AROUND_ROTATION_AXIS:
    gamma = ( ( (double)xpix + (double)pd->channel_offset + 
		( (double)xpoint + 0.5 ) / (double)pd->points_xy )
	      / (double)pd->pixel_x * 2.0 - 1.0 ) * pd->fanangle;

    ray_detector[0] =  pd->source2det * tan(gamma); //maja 2005-11

    ray_detector[2] = ( -0.5 * pd->length_y +
			( (double)ypix + ((double)ypoint + 0.5) 
			  / (double)pd->points_xy ) 
			* pd->pix_lgth_y + pd->skew * gamma ) 
                        / (cos(gamma) * cos(gamma));
    break;
  }
} 


/*******************************************************************
 * Calculation of intersection with ellipses, cylinders, boxes and *
 * voxel volumes.                                                  *
 *******************************************************************/
double calc_intersection(OBJECT *po, POINT pt_src, POINT pt_det, SETUP *su)
{
  POINT hit1, hit2;
  POINT pt_src_obj, pt_det_obj;
  double leng;
  char isec;

  /* transform detector and source to object space */
  trfm(pt_det, po->inv_obj, pt_det_obj);  
  trfm(pt_src, po->inv_obj, pt_src_obj);

  switch(po->kind_of_obj) { 		/* Intersection? */
  case OBJECT_ELLIPSOID:
    isec = calc_intersection_ellip(pt_src_obj, pt_det_obj, hit1, hit2);
    break;
  case OBJECT_CYLINDER:
    isec = calc_intersection_cylind(pt_src_obj, pt_det_obj, hit1, hit2);
    break;
  case OBJECT_BOX:
    isec = calc_intersection_box(pt_src_obj, pt_det_obj, hit1, hit2);
    break;
  case OBJECT_VOXELS:
    isec = calc_intersection_box(pt_src_obj, pt_det_obj, hit1, hit2);
    if (isec) {
      switch(su->kind_of_projgen) { /* Projgen type? */ //maja 2005
      case KOHLER:
	po->dens = lineint_in_voxvol_kohler(
		   po->voxeldata, po->Nvox, hit1, hit2);
	break;
      case SIDDON:
	po->dens = lineint_in_voxvol_siddon(
		   po->voxeldata, po->Nvox, hit1, hit2);
	break;
      case JOSEPH:
	po->dens = lineint_in_voxvol_joseph(
                   po->voxeldata, po->Nvox, hit1, hit2);
	break;
      case JOSEPHNN:
	po->dens = lineint_in_voxvol_josephnn(
                   po->voxeldata, po->Nvox, hit1, hit2);
	break;
      }
    }
    break;
  }
  
  /* transform back from object space */
  if (isec)
    leng = length(hit1, hit2, po);
  else
    leng = 0.0;

  return(leng);
}

/****************************************************************************
 * Calculates the intersection of a transformed ray and a transformed       *
 * ellipsoid (= generic sphere).                                            *
 *                                                                          *
 * author: Jens Mueller-Merbach                                      nov 95 *
 *                                                                          *
 * Input: ray that is transformed into 'sphere-space' for object number     *
 * obj_nr. (Because each object is transformed into the sphere-space by     *
 * another transformation.)                                                 *
 * Return: 0 if no intersection                                             *
 *         1 if intersection                                                *
 * BUGS: If the source or the detector are lying in an object, this routine *
 * still calculates the length of the whole intersection of the (mathemati- *
 * cal) ray instead of only a part of it. If the source is e. g. in the     *
 * center of a sphere, the intersection should be half the diameter, but it *
 * is the whole diameter.                                                   *
 ****************************************************************************/
char calc_intersection_ellip (POINT src, /* transformed source position */
			      POINT det, /* transf. point where ray hits det. */
			      POINT pt1,
			      POINT pt2) /* points where ray hits sphere */
{
  double dir[3];		/* ray direction */
  double dir2 = 0.0;		/* length of ray dir. vector ^2 */
  double src2 = 0.0;		/* length of vector to source ^2 */
  double scp = 0.0;		/* scalar product: source * direction of ray */
  double discriminant = 0.0;
  double te, dis, te1, te2;     
  int i;                        

  /*----------------------------------------------*
   * Calculate direction of ray and its length ^2 *
   * scalarproduct source*direction 
   * and length of vector to source               *
   *----------------------------------------------*/
  for (i=0; i<3; i++) {
    dir[i] = det[i]-src[i];
    dir2 += dir[i]*dir[i];
    src2 += src[i]*src[i];
    scp += src[i]*dir[i];
  }

  /*-------------------------------------------------------*
   * Calculate discriminant: if <=0 --> no intersection    *
   *-------------------------------------------------------*/
  discriminant = scp*scp - src2*dir2 + dir2;

  if (discriminant <= 0.0) 
    return 0;

  /*--------------------------------------------------------------------------*
   * Calculate the two points where ray hits sphere; te, dis introduced to    *
   * shorten calc. time                                                       *
   *--------------------------------------------------------------------------*/
  te = -scp/dir2;
  dis = sqrt(discriminant)/dir2;
  te1 = te + dis;
  te2 = te - dis;
  for (i=0; i<3; i++) {
    pt1[i] = src[i] + te1 * dir[i];
    pt2[i] = src[i] + te2 * dir[i];
  }

  return 1;
}


/****************************************************************************
 * calculates the intersection of a transformed ray and a transformed       *
 * cylinder  (= generic cylinder)                                           *
 *                                                                          *
 * author: Jens Mueller-Merbach                                  jan 09, 96 *
 *                                                                          *
 * Input: ray that is transformed into 'sphere-space' for object number     *
 * obj_nr. (Because each object is transformed into the sphere-space by     *
 * another transformation.)                                                 *
 * Return: 0 if no intersection                                             *
 *         1 if intersection                                                *
 * BUGS: If the source or the detector is lying in an object, this routine  *
 * still calculates the length of the whole intersection of the (mathemati- *
 * cal) ray instead of only a part of it. If the source is e. g. in the     *
 * center of a cylinder, the intersection should be half the diameter, but  *
 * it is calculated as the whole diameter.                                  *
 ****************************************************************************/
char calc_intersection_cylind (
     POINT ray_source_tf,	/* transformed source position= s */
     POINT ray_detector_tf, /* transf. point where ray hits det.= d */
     POINT hit_pt1, POINT hit_pt2) /* points where ray hits cylinder */
{
  double ray_direction[3];      /* ray direction: c = d - s */
  double ray_dir_len_xy = 0.0;	/* cx*cx + cy*cy */
  double ray_source_len_xy = 0.0; /* sx*sx + sy*sy */
  double source_times_dir = 0.0; /* sx*cx + sy*cy */
  double two_sx_sy_cx_cy = 2.0;	/* 2*sx*sy*cx*cy */
  double discriminant = 0.0;
  double te, dis, te1, te2;	/* dummies */
  double hit[2][3];		/* hitting points */
  int i;			/* counter */

  /*----------------------------------------------*
   * Calculate the direction of the ray           *
   *----------------------------------------------*/
  for (i=0; i<3; i++) 
    ray_direction[i] = ray_detector_tf[i]-ray_source_tf[i];

  /*--------------------------------------------------------------------------*
   * Calculate some values that are needed to find the cylinder hitting points*
   *--------------------------------------------------------------------------*/
  for (i=0; i<=1; i++)  /* i<=1 for we only want the x- and y-coordinates */
    {
      ray_dir_len_xy += ray_direction[i]*ray_direction[i];
      ray_source_len_xy += ray_source_tf[i]*ray_source_tf[i];
      source_times_dir += ray_source_tf[i]*ray_direction[i];
      two_sx_sy_cx_cy *= ray_source_tf[i]*ray_direction[i];
    }

  /*----------------------------------------------------------*
   * ray is absolutely parallel to the cylinder coat?         *
   * in this case, the discriminant (see below) is undefined. *
   *----------------------------------------------------------*/
  if ((fabs((double) ray_direction[0]) < 0.000000001) &&
      (fabs((double) ray_direction[1]) < 0.000000001)) {
    if (ray_source_len_xy <= 1.0) {
      hit_pt1[0]=hit_pt2[0]=ray_source_tf[0];
      hit_pt1[1]=hit_pt2[1]=ray_source_tf[1];
      hit_pt1[2]=1.0;
      hit_pt1[2]=-1.0;
      return 1;
    } else
      return 0;
    }

  /*----------------------------------------------*
   * Calculate the discriminant                   *
   *----------------------------------------------*/
  discriminant = (source_times_dir*source_times_dir)/ray_dir_len_xy;
  discriminant += (1.0 - ray_source_len_xy);

  if (discriminant > 0.0)
    {
      /*------------------------------------------------*
       * ray hits coat of cylinder: calc hitting points *
       *------------------------------------------------*/
      discriminant /= ray_dir_len_xy;
      te = -source_times_dir/ray_dir_len_xy;
      dis = sqrt((double)discriminant);
      /*-------------------------*
       * hit[0][2] > hit[1][2] ! *
       *-------------------------*/
      if (ray_direction[2] < 0.0)
	{
	  te1 = te - dis;
	  te2 = te + dis;
	}
      else
	{
	  te1 = te + dis;
	  te2 = te - dis;
	}
      for (i=0; i<3; i++)
	{
	  hit[0][i] = ray_source_tf[i] + te1 * ray_direction[i];
	  hit[1][i] = ray_source_tf[i] + te2 * ray_direction[i];
	}
      /*-----------------------------------------------------*
       * does hit[0] lie between top and bottom of cylinder? *
       *-----------------------------------------------------*/
      if (fabs((double) hit[0][2]) < 1.0)
	{ 
	  for(i=0;i<=2;i++) hit_pt1[i] = hit[0][i];
	  if (fabs((double) hit[1][2]) < 1.0)
	    {
	      /*-----------------------------*
	       * ray hits the coat two times *
	       *-----------------------------*/
	      for(i=0;i<=2;i++) hit_pt2[i] = hit[1][i];
	      return 1;
	    }
	  else
	    {
	      /*--------------------------------------------------*
	       * ray hits the coat and the bottom of the cylinder *
	       *--------------------------------------------------*/
	      te2 = - (1.0+ray_source_tf[2]) / ray_direction[2];
	      for (i=0; i<=2; i++) hit_pt2[i] = ray_source_tf[i] 
				     + te2 * ray_direction[i];
	      return 1;
	    }
	}
      else /* hit[0] does not lie between top and bottom */
	{
	  if (hit[0][2] >= 1.0)
	    {
	      /*-------------------------------------------------*
	       * point hit[0] lies above the top of the cylinder *
	       *-------------------------------------------------*/
	      if (fabs((double) hit[1][2]) < 1.0)
		{
		  /*-----------------------------------------------*
		   * ray hits the coat and the top of the cylinder *
		   *-----------------------------------------------*/
		  for(i=0;i<=2;i++) hit_pt2[i] = hit[1][i];
		  te1 = (1.0-ray_source_tf[2]) / ray_direction[2];
		  for (i=0; i<=2; i++) hit_pt1[i] = ray_source_tf[i] 
					 + te1 * ray_direction[i];
		  return 1;
		}
	      else
		{
		  if (hit[1][2] < 1.0)
		    {
		      /*-------------------------------------------------*
		       * ray hits the bottom and the top of the cylinder *
		       *-------------------------------------------------*/
		      te1 = (1.0-ray_source_tf[2]) / ray_direction[2];
		      for(i=0; i<=2; i++) hit_pt1[i] = ray_source_tf[i] 
					    + te1 * ray_direction[i];
		      te2 = - (1.0+ray_source_tf[2]) / ray_direction[2];
		      for(i=0; i<=2; i++) hit_pt2[i] = ray_source_tf[i] 
					    + te2 * ray_direction[i];
		      return 1;
		    }
		  else
		    return 0;
		}
	    }
	  else /* hit[0] lies below the bottom -> no intersection */
	    return 0;
	}
    }
  else 
    /*---------------------------------------*
     * discriminant < 0.0 -> no intersection *
     * ray does not hit coat of cylinder     *
     *---------------------------------------*/
    return 0;
}

/****************************************************************************
 * calculates the intersection of a transformed ray and a transformed       *
 * box                                                                      *
 *                                                                          *
 * author: Henrik Turbell                                            jun 99 *
 *                                                                          *
 * Input: ray that is transformed into 'cube-space' for object number       *
 * obj_nr. (Because each object is transformed into the cube-space by       *
 * another transformation.)                                                 *
 * Return: 0 if no intersection                                             *
 *         1 if intersection                                                *
 * BUGS: If the source or the detector are lying in an object, this routine *
 * still calculates the length of the whole intersection of the (mathemati- *
 * cal) ray instead of only a part of it. If the source is e. g. in the     *
 * center of a sphere, the intersection should be half the diameter, but it *
 * is the whole diameter.                                                   *
 ****************************************************************************/
char calc_intersection_box (
     POINT ray_source_tf,	/* transformed source position */
     POINT ray_detector_tf,     /* transformed point where ray hits detector */
     POINT hit_pt1,
     POINT hit_pt2)            /* points where ray hits box */
{
  double t;
  double x, y, z;
  double x1, x2, y1, y2, z1, z2;
  double hit[6][3];  /* Max 6 intersections, but use only first two */
  int Nintersect = 0;
  
  x1 = ray_source_tf[0];
  x2 = ray_detector_tf[0];
  y1 = ray_source_tf[1];
  y2 = ray_detector_tf[1];
  z1 = ray_source_tf[2];
  z2 = ray_detector_tf[2];
 
  /* printf("(%f, %f, %f) -> (%f, %f, %f)\n", x1, y1, z1, x2, y2, z2); */

  if (fabs(x1 - x2) > 0.00001)
    {
      x = 1.0;
      t = (x - x2) / (x1 - x2);             
      y = t * y1 + (1.0 -  t) * y2;
      if (fabs(y) <= 1.0)
	{
	  z = t * z1 + (1.0 -  t) * z2;
	  if (fabs(z) <= 1.0)
	    {
	      hit[Nintersect][0] = x;
	      hit[Nintersect][1] = y;
	      hit[Nintersect][2] = z;
	      Nintersect++;
	    }
	}
      
      x = -1.0;
      t = (x - x2) / (x1 - x2);
      y = t * y1 + (1.0 -  t) * y2;
      if (fabs(y) <= 1.0)
	{
	  z = t * z1 + (1.0 -  t) * z2;
	  if (fabs(z) <= 1.0)
	    {
	      hit[Nintersect][0] = x;
	      hit[Nintersect][1] = y;
	      hit[Nintersect][2] = z;
	      Nintersect++;
	    }
	}      
    }

  if (fabs(y1 - y2) > 0.00001)
    {
      y = 1.0;
      t = (y - y2) / (y1 - y2);
      x = t * x1 + (1.0 -  t) * x2;
      if (fabs(x) <= 1.0)
	{
	  z = t * z1 + (1.0 -  t) * z2;
	  if (fabs(z) <= 1.0)
	    {
	      hit[Nintersect][0] = x;
	      hit[Nintersect][1] = y;
	      hit[Nintersect][2] = z;
	      Nintersect++;
	    }
	}
      
      y = -1.0;
      t = (y - y2) / (y1 - y2);
      x = t * x1 + (1.0 -  t) * x2;
      if (fabs(x) <= 1.0)
	{
	  z = t * z1 + (1.0 -  t) * z2;
	  if (fabs(z) <= 1.0)
	    {
	      hit[Nintersect][0] = x;
	      hit[Nintersect][1] = y;
	      hit[Nintersect][2] = z;
	      Nintersect++;
	    }
	}
    }
  
  if (fabs(z1 - z2) > 0.00001)
    {
      z = 1.0;
      t = (z - z2) / (z1 - z2);
      x = t * x1 + (1.0 -  t) * x2;
      if (fabs(x) <= 1.0)
	{
	  y = t * y1 + (1.0 -  t) * y2;
	  if (fabs(y) <= 1.0)
	    {
	      hit[Nintersect][0] = x;
	      hit[Nintersect][1] = y;
	      hit[Nintersect][2] = z;
	      Nintersect++;
	    }
	}
      
      z = -1.0;
      t = (z - z2) / (z1 - z2);
      x = t * x1 + (1.0 -  t) * x2;
      if (fabs(x) <= 1.0)
	{
	  y = t * y1 + (1.0 -  t) * y2;
	  if (fabs(y) <= 1.0)
	    {
	      hit[Nintersect][0] = x;
	      hit[Nintersect][1] = y;
	      hit[Nintersect][2] = z;
	      Nintersect++;
	    }
	}
    }

  if (Nintersect < 2)
    return 0;

  hit_pt1[0] = hit[0][0];
  hit_pt1[1] = hit[0][1];
  hit_pt1[2] = hit[0][2];

  hit_pt2[0] = hit[1][0];
  hit_pt2[1] = hit[1][1];
  hit_pt2[2] = hit[1][2];
 
  return 1;
}

/****************************************************************************
 * Checks if a given point (coordintes calculated in subroutine voxeli-     *
 * zation) is located in a given ellipsoid. If so, it returns the value of  *
 * the density of the object, otherwise it returns zero.                    *
 *                                                                          *
 * author: Jens Mueller-Merbach                                  jan 12, 96 *
 *                                                                          *
 * Input: phantom data (to_obj, nr_of_obj), voxel coordinates               *
 * Output: ./.                                                              *
 * Return: density of the object if point is in it, zero otherwise.         *
 ****************************************************************************/
double p_in_ellip (OBJECT *po,	/* object data */
		    double *pos, /* coordinates of the point */
		    double *p_tf) /* transformed point */
{
  double den=0.0;     /* density */
  double rad;
    
  /*--------------------------------------------*
   * transform the point into the sphere system * 
   *--------------------------------------------*/
  trfm(pos, po->inv_obj, p_tf);

  rad = p_tf[0]*p_tf[0] + p_tf[1]*p_tf[1] + p_tf[2]*p_tf[2];
  /* rad = (double) sqrt((double)rad); */
  if (rad <= 1.0) den = po->dens;

  return den;
}  


/****************************************************************************
 * Checks if a given point (coordintes calculated in subroutine voxeli-     *
 * zation) is located in a given cylinder.  If so, it returns the value of  *
 * the density of the object, otherwise it returns zero.                    *
 *                                                                          *
 * author: Jens Mueller-Merbach                                  jan 12, 96 *
 *                                                                          *
 * Input: phantom data (to_obj, nr_of_obj), voxel coordinates               *
 * Output: ./.                                                              *
 * Return: density of the object if point is in it, zero otherwise.         *
 ****************************************************************************/
double p_in_cylind(OBJECT *po,
		   double *pos,       /* coordinates of the point */
		   double *p_tf)      /* transformed point */
{
  double rad;
  double den=0.0;      /* density */

  /*------------------------------------------------------*
   * transform the point into the generic cylinder system * 
   *------------------------------------------------------*/
  trfm(pos, po->inv_obj, p_tf);

  if (fabs(p_tf[2]) <= 1.0)
    {
      rad = p_tf[0]*p_tf[0] + p_tf[1]*p_tf[1];
      /* rad = (double) sqrt((double)rad); */
      if (rad <= 1.0) den = po->dens;
    }
  return den;
}  

/****************************************************************************
 * Checks if a given point (coordintes calculated in subroutine voxeli-     *
 * zation) is located in a given box. If so, it returns the value of        *
 * the density of the object, otherwise it returns zero.                    *
 *                                                                          *
 * author: Henrik Turbell                                        jun 30, 99 *
 *                                                                          *
 * Input: phantom data (to_obj, nr_of_obj), voxel coordinates               *
 * Output: ./.                                                              *
 * Return: density of the object if point is in it, zero otherwise.         *
 ****************************************************************************/
double p_in_box (OBJECT *po,	/* pointer to object data */
		 double *pos,	/* coordinates of the point */
		 double *p_tf)	/* transformed point */
{
  double den=0.0;     /* density */   
    
  /*--------------------------------------------*
   * transform the point into the sphere system * 
   *--------------------------------------------*/
  trfm(pos, po->inv_obj, p_tf);

  if ( (fabs(p_tf[0])<=1.0) && (fabs(p_tf[1])<=1.0) && (fabs(p_tf[2])<=1.0) )
    den = po->dens;
   
  return den;
}  

/****************************************************************************
 * Line intergration using Simpson's rule.                                  *
 * Original idea by Thomas Khler                                           *
 *                                                                          *
 * author: Henrik Turbell                                            jun 99 *
 *                                                                          *
 * Input: voxeldata with margin                                             *
 *        size in 3 dimension excluding margin                              *
 *        start and end points given on "unit" cube -1...1                  *
 * Output: none                                                             *
 * Return: average density along the ray                                    *
 ****************************************************************************/
double lineint_in_voxvol_kohler(float* voxeldata, int Nvox[3],
				double p1[3], double p2[3])
{
  double x1, x2, y1, y2, z1, z2;
  double dx, dy, dz;
  double abs_dx, abs_dy, abs_dz;
  double half_dx, half_dy, half_dz;
  char pos_dx, pos_dy, pos_dz;
  double inv_dx, inv_dy, inv_dz;
  int Nx, Ny, Nz, Nxy, Nxyz;
  int Mx, My, Mz, Mxy, Mxyz, Mxy_Mx;
  int Mx_1, Mxy_1, Mxy_Mx_1;   
  float* pos;
  double dens, contrib;
  double t;
  double x, y, z;
  double xm, ym, zm;
  double delta_t, deltat_x, deltat_y, deltat_z;
  double old_bilinear, new_bilinear, trilinear;
  double wx, wy, wz, wx_, wy_, wz_;
  int x_floor, y_floor, z_floor;

  /* Sizes without margin */
  /*----------------------*/
  Nx = Nvox[0];
  Ny = Nvox[1];
  Nz = Nvox[2];
  Nxy = Nx * Ny;
  Nxyz = Nxy * Nz;
  
  /* Sizes with margin */
  /*-------------------*/
  Mx = Nx + 2 * VOXMARG;
  My = Ny + 2 * VOXMARG;
  Mz = Nz + 2 * VOXMARG;
  Mx_1 = Mx + 1;
  Mxy = Mx * My;
  Mxy_1 = Mx * My + 1; 
  Mxy_Mx = Mxy + Mx;
  Mxy_Mx_1 = Mxy + Mx + 1;
  Mxyz = Mxy * Mz;

  /* Scale from continous unit [-1..1] to    */
  /* discrete voxelindex [0..Nx-1+2*VOXMARG] */
  /* -1 => -0.5 + VOXMARG                    */
  /* 0 => Nx / 2 - 0.5 + VOXMARG             */
  /* 1 => Nx - 0.5 + VOXMARG                 */
  /*-----------------------------------------*/
  x1 = (p1[0] + 1.0) / 2.0 * Nx - 0.5 + VOXMARG;
  x2 = (p2[0] + 1.0) / 2.0 * Nx - 0.5 + VOXMARG;
  y1 = (p1[1] + 1.0) / 2.0 * Ny - 0.5 + VOXMARG;
  y2 = (p2[1] + 1.0) / 2.0 * Ny - 0.5 + VOXMARG;
  z1 = (p1[2] + 1.0) / 2.0 * Nz - 0.5 + VOXMARG;
  z2 = (p2[2] + 1.0) / 2.0 * Nz - 0.5 + VOXMARG;

  dx = x2 - x1;
  dy = y2 - y1;
  dz = z2 - z1;

  half_dx = dx / 2.0;
  half_dy = dy / 2.0;
  half_dz = dz / 2.0;

  abs_dx = fabs(dx);
  abs_dy = fabs(dy);
  abs_dz = fabs(dz);

  inv_dx = (abs_dx > SMALL_NUMBER) ? 1.0 / dx : 
    ((dx >= 0.0) ? BIG_NUMBER : -BIG_NUMBER);
  inv_dy = (abs_dy > SMALL_NUMBER) ? 1.0 / dy : 
    ((dy >= 0.0) ? BIG_NUMBER : -BIG_NUMBER);
  inv_dz = (abs_dz > SMALL_NUMBER) ? 1.0 / dz : 
    ((dz >= 0.0) ? BIG_NUMBER : -BIG_NUMBER);

  pos_dx = (dx < 0) ? 0 : 1;
  pos_dy = (dy < 0) ? 0 : 1;
  pos_dz = (dz < 0) ? 0 : 1;

  t = 0;
  dens = 0.0;

  x = x1;
  y = y1;
  z = z1;

  /* calculate tri-linear entrance point as first bi-linear value */
  /*--------------------------------------------------------------*/
  x_floor = floor(x);
  y_floor = floor(y);
  z_floor = floor(z);
  
  wx = x - x_floor;
  wy = y - y_floor;
  wz = z - z_floor;		    
  
  wx_ = 1.0 - wx;
  wy_ = 1.0 - wy;
  wz_ = 1.0 - wz;
  
  pos = voxeldata + z_floor * Mxy + y_floor * Mx + x_floor;

  old_bilinear = wz_ * ( wy_ * (wx_ * *(pos) + wx * *(pos + 1)) +
			 wy * (wx_ * *(pos + Mx) + wx * *(pos + Mx_1))) +
    wz * ( wy_ * (wx_ * *(pos + Mxy) + wx * *(pos + Mxy_1)) +
	   wy * (wx_ * *(pos + Mxy_Mx) + wx * *(pos + Mxy_Mx_1)));  
  
  /* Move stepwise from startpoint, where t=0, to endpoint, where t=1. */
  /*-------------------------------------------------------------------*/
  while (t < 1.0)
  {        
    deltat_x = pos_dx ? (1-x+floor(x)) * inv_dx : (-1-x+ceil(x)) * inv_dx;
    deltat_y = pos_dy ? (1-y+floor(y)) * inv_dy : (-1-y+ceil(y)) * inv_dy;
    deltat_z = pos_dz ? (1-z+floor(z)) * inv_dz : (-1-z+ceil(z)) * inv_dz;
    
    if ((deltat_x < deltat_y) && (deltat_x < deltat_z))
    {
      /* Move forward to yz-wall and perform bilinear interpolation */
      /*------------------------------------------------------------*/
      delta_t = deltat_x;
      
      x = pos_dx ? floor(x + 1.0) : ceil(x - 1.0);
      y += delta_t * dy;
      z += delta_t * dz;
	  
      x_floor = floor(x);
      y_floor = floor(y);
      z_floor = floor(z);	 
	  
      pos = voxeldata + 
	z_floor * Mxy + y_floor * Mx + x_floor; /* position in voxeldata */
      
      wy = y - y_floor;
      wz = z - z_floor;
	  
      new_bilinear = (1.0 - wz) * (*(pos) * (1.0 - wy) + *(pos + Mx) * wy) +
	wz * (*(pos + Mxy) * (1.0 - wy) + *(pos + Mxy_Mx) * wy);

      if (pos_dx)
      {	/* step backwards to prepare for action at point in between */
	x_floor--;
	pos--;  	  	  
      }
    }	  	          
    else if (deltat_y < deltat_z)
    {
      /* Move forward to xz-wall and perform bilinear interpolation */
      /*------------------------------------------------------------*/
      delta_t = deltat_y;
      
      x += delta_t * dx;
      y = pos_dy ? floor(y + 1.0) : ceil(y - 1.0);	     
      z += delta_t * dz;
	  
      x_floor = floor(x);
      y_floor = floor(y);
      z_floor = floor(z);
	  
      pos = voxeldata + 
	z_floor * Mxy + y_floor * Mx + x_floor; /* position in voxeldata */
      
      wx = x - x_floor;
      wz = z - z_floor;
	  
      new_bilinear = (1.0 - wz) * (*(pos) * (1.0 - wx) + *(pos + 1) * wx) +
	wz * (*(pos + Mxy) * (1.0 - wx) + *(pos + Mxy_1) * wx);	  	 
	  
      if (pos_dy)
      {	/* step backwards to prepare for action at point in between */
	y_floor--;
	pos -= Mx;	 		      	  	 
      }
    } 
    else 
    {
      /* Move forward to xy-wall and perform bilinear interpolation */
      /*------------------------------------------------------------*/
      delta_t = deltat_z;
      
      x += delta_t * dx;
      y += delta_t * dy;
      z = pos_dz ? floor(z + 1.0) : ceil(z - 1.0);	     
      
      x_floor = floor(x);
      y_floor = floor(y);
      z_floor = floor(z);

      pos = voxeldata + 
	z_floor * Mxy + y_floor * Mx + x_floor; /* position in voxeldata */
      
      wx = x - x_floor;
      wy = y - y_floor;
      
      new_bilinear = (1.0 - wy) * (*(pos) * (1.0 - wx) + *(pos + 1) * wx) +
	wy * (*(pos + Mx) * (1.0 - wx) + *(pos + Mx_1) * wx);	
      
      if (pos_dz)
      {	/* step backwards to prepare for action at point in between */
	z_floor--;
	pos -= Mxy;	  		  	 	      
      }
    }	  	          
  
    /* calculate trilinear interpolation at point in between */
    /*-------------------------------------------------------*/
    xm = x - delta_t * half_dx;
    ym = y - delta_t * half_dy;
    zm = z - delta_t * half_dz;	  	  
      
    wx = xm - x_floor;
    wy = ym - y_floor;
    wz = zm - z_floor;		    
      
    wx_ = 1.0 - wx;
    wy_ = 1.0 - wy;
    wz_ = 1.0 - wz;
      
    trilinear = wz_ * ( wy_ * (wx_ * *(pos) + wx * *(pos + 1)) +
			wy * (wx_ * *(pos + Mx) + wx * *(pos + Mx_1))) +
      wz * ( wy_ * (wx_ * *(pos + Mxy) + wx * *(pos + Mxy_1)) +
	     wy * (wx_ * *(pos + Mxy_Mx) + wx * *(pos + Mxy_Mx_1)));
    
    t += delta_t;
    
    contrib = old_bilinear + 4.0 * trilinear + 
      new_bilinear; /* Simpson's rule without normaliation */
    
    dens += delta_t * contrib;
    
    old_bilinear = new_bilinear;
  }          
  
  dens -= (t-1.0) * contrib; /* remove any surplus contribution */
  
  dens /= 6.0; /* normalization in Simpson's rule */
  
  return dens;
}

/****************************************************************************
 * Line intergration using Siddons approach.                                *
 *                                                                          *
 * Written by Maria Magnusson Seger, 2005-12,                               *
 * inspired by code from Henrik Turbell                                     *
 *                                                                          *
 * Input: voxeldata with margin                                             *
 *        size in 3 dimension excluding margin                              *
 *        start and end points given on "unit" cube -1...1                  *
 * Output: none                                                             *
 * Return: average density along the ray                                    *
 ****************************************************************************/
double lineint_in_voxvol_siddon(float* voxeldata, int Nvox[3],
				double p1[3], double p2[3])
{
  double x1, x2, y1, y2, z1, z2;
  double dx, dy, dz;
  double abs_dx, abs_dy, abs_dz;
  double half_dx, half_dy, half_dz;
  char pos_dx, pos_dy, pos_dz;
  double inv_dx, inv_dy, inv_dz;
  int Nx, Ny, Nz, Nxy, Nxyz;
  int Mx, My, Mz, Mxy, Mxyz, Mxy_Mx;
  int Mx_1, Mxy_1, Mxy_Mx_1;   
  float* pos;
  double dens;
  double t;
  double x, y, z;
  double xm, ym, zm;
  double delta_t, deltat_x, deltat_y, deltat_z;

  /* Sizes without margin */
  /*----------------------*/
  Nx = Nvox[0];
  Ny = Nvox[1];
  Nz = Nvox[2];
  Nxy = Nx * Ny;
  Nxyz = Nxy * Nz;
  
  /* Sizes with margin */
  /*-------------------*/
  Mx = Nx + 2 * VOXMARG;
  My = Ny + 2 * VOXMARG;
  Mz = Nz + 2 * VOXMARG;
  Mx_1 = Mx + 1;
  Mxy = Mx * My;
  Mxy_1 = Mx * My + 1; 
  Mxy_Mx = Mxy + Mx;
  Mxy_Mx_1 = Mxy + Mx + 1;
  Mxyz = Mxy * Mz;

  /* Scale from continous unit [-1..1] to    */
  /* discrete voxelindex [0..Nx-1+2*VOXMARG] */
  /* -1 => VOXMARG                           */
  /* 0 => Nx / 2 + VOXMARG                   */
  /* 1 => Nx + VOXMARG                       */
  /*-----------------------------------------*/
  x1 = (p1[0] + 1.0) / 2.0 * Nx + VOXMARG;
  x2 = (p2[0] + 1.0) / 2.0 * Nx + VOXMARG;
  y1 = (p1[1] + 1.0) / 2.0 * Ny + VOXMARG;
  y2 = (p2[1] + 1.0) / 2.0 * Ny + VOXMARG;
  z1 = (p1[2] + 1.0) / 2.0 * Nz + VOXMARG;
  z2 = (p2[2] + 1.0) / 2.0 * Nz + VOXMARG;

  dx = x2 - x1;
  dy = y2 - y1;
  dz = z2 - z1;

  half_dx = dx / 2.0;
  half_dy = dy / 2.0;
  half_dz = dz / 2.0;

  abs_dx = fabs(dx);
  abs_dy = fabs(dy);
  abs_dz = fabs(dz);

  inv_dx = (abs_dx > SMALL_NUMBER) ? 1.0 / dx : 
    ((dx >= 0.0) ? BIG_NUMBER : -BIG_NUMBER);
  inv_dy = (abs_dy > SMALL_NUMBER) ? 1.0 / dy : 
    ((dy >= 0.0) ? BIG_NUMBER : -BIG_NUMBER);
  inv_dz = (abs_dz > SMALL_NUMBER) ? 1.0 / dz : 
    ((dz >= 0.0) ? BIG_NUMBER : -BIG_NUMBER);

  pos_dx = (dx < 0) ? 0 : 1;
  pos_dy = (dy < 0) ? 0 : 1;
  pos_dz = (dz < 0) ? 0 : 1;

  t = 0;
  dens = 0.0;

  x = x1;
  y = y1;
  z = z1;

  /* Move stepwise from startpoint, where t=0, to endpoint, where t=1. */
  /*-------------------------------------------------------------------*/
  while (t < 1.0)
  {        
    deltat_x = pos_dx ? (1-x+floor(x)) * inv_dx : (-1-x+ceil(x)) * inv_dx;
    deltat_y = pos_dy ? (1-y+floor(y)) * inv_dy : (-1-y+ceil(y)) * inv_dy;
    deltat_z = pos_dz ? (1-z+floor(z)) * inv_dz : (-1-z+ceil(z)) * inv_dz;
    
    if ((deltat_x < deltat_y) && (deltat_x < deltat_z))
    {
      /* Move forward to yz-wall */
      /*-------------------------*/
      delta_t = deltat_x;
      
      x = pos_dx ? floor(x + 1.0) : ceil(x - 1.0);
      y += delta_t * dy;
      z += delta_t * dz;
    }	  	          
    else if (deltat_y < deltat_z)
    {
      /* Move forward to xz-wall */
      /*-------------------------*/
      delta_t = deltat_y;
      
      x += delta_t * dx;
      y = pos_dy ? floor(y + 1.0) : ceil(y - 1.0);	     
      z += delta_t * dz;
    } 
    else 
    {
      /* Move forward to xy-wall */
      /*-------------------------*/
      delta_t = deltat_z;
      
      x += delta_t * dx;
      y += delta_t * dy;
      z = pos_dz ? floor(z + 1.0) : ceil(z - 1.0);	     
    }	  	          
  
    /* register voxel value at a point in between */
    /*--------------------------------------------*/
    xm = x - delta_t * half_dx;
    ym = y - delta_t * half_dy;
    zm = z - delta_t * half_dz;	  	  
      
    pos = voxeldata + (int) (floor(zm) * Mxy + floor(ym) * Mx + floor(xm));
    t += delta_t;
    dens += delta_t * *pos;
  }          
  
  dens -= (t-1.0) * *pos; /* remove any surplus contribution */

  return dens;
}

/****************************************************************************
 * Line intergration using Joseph's approach (bilinear interp).             *
 *                                                                          *
 * author: Henrik Turbell                                            oct 00 *
 * Updated by Maria Magnusson Seger 2005-12-19                              *
 *                                                                          *
 * Input: voxeldata with margin                                             *
 *        size in 3 dimension excluding margin                              *
 *        start and end points given on "unit" cube -1...1                  *
 * Output: none                                                             *
 * Return: average density along the ray                                    *
 ****************************************************************************/
double lineint_in_voxvol_joseph(float* voxeldata, int Nvox[3],
				double p1[3], double p2[3])
{
  double x1, x2, y1, y2, z1, z2;
  double dx, dy, dz;
  double abs_dx, abs_dy, abs_dz;  
  int Nx, Ny, Nz, Nxy, Nxyz;
  int Mx, My, Mz, Mxy, Mxy_Mx, Mxy_1;
  double dens = 0.0;
  double t;
  double x, y, z;     
  float* pos;    
  double dummy, dydx, dzdx, dxdy, dzdy;
  double wx, wy, wz;

  /* Sizes without margin */
  /*----------------------*/
  Nx = Nvox[0];
  Ny = Nvox[1];
  Nz = Nvox[2];
  Nxy = Nx * Ny;
  Nxyz = Nxy * Nz;
  
  /* Sizes with margin */
  /*-------------------*/
  Mx = Nx + 2 * VOXMARG;
  My = Ny + 2 * VOXMARG;
  Mz = Nz + 2 * VOXMARG; 
  Mxy = Mx * My;
  Mxy_Mx = Mxy + Mx;
  Mxy_1 = Mxy + 1;
 
  /* Scale from continous unit [-1..1] to    */
  /* discrete voxelindex [0..Nx-1+2*VOXMARG] */
  /* -1 => -0.5 + VOXMARG                    */
  /* 0 => Nx / 2 - 0.5 + VOXMARG             */
  /* 1 => Nx - 0.5 + VOXMARG                 */
  /*-----------------------------------------*/
  x1 = (p1[0] + 1.0) / 2.0 * Nx - 0.5 + VOXMARG;
  x2 = (p2[0] + 1.0) / 2.0 * Nx - 0.5 + VOXMARG;
  y1 = (p1[1] + 1.0) / 2.0 * Ny - 0.5 + VOXMARG;
  y2 = (p2[1] + 1.0) / 2.0 * Ny - 0.5 + VOXMARG;
  z1 = (p1[2] + 1.0) / 2.0 * Nz - 0.5 + VOXMARG;
  z2 = (p2[2] + 1.0) / 2.0 * Nz - 0.5 + VOXMARG;

  dx = x2 - x1;
  dy = y2 - y1;
  dz = z2 - z1; 
  
  abs_dx = fabs(dx);
  abs_dy = fabs(dy);
  abs_dz = fabs(dz); 
  
  /* calculate the line integral either in principal direction x or y */
  /*------------------------------------------------------------------*/
  if ((abs_dx >= abs_dy) && (abs_dx >= abs_dz)) //maja 2005
  {
    /* principal direction: x */
    if (x2 < x1)
    {
      /* switch start and stop point */
      dummy = x1; x1 = x2; x2 = dummy;
      dummy = y1; y1 = y2; y2 = dummy;
      dummy = z1; z1 = z2; z2 = dummy;
      
      dx = x2 - x1;
      dy = y2 - y1;
      dz = z2 - z1;        
    }
           
    dydx = dy / dx;
    dzdx = dz / dx;
    
    t = ceil(x1) - x1;
    
    /* (x,y,z) is the location of the first interpolation location */
    x = floor(x1 + t + 0.1);
    y = y1 + t * dydx;
    z = z1 + t * dzdx;
    
    dens = 0.0;
    
    /* step trough the volume and accumulate the line integral */
    while(x < x2)		
    {      
      pos = voxeldata + (int) (floor(z) * Mxy + floor(y) * Mx + floor(x));
      wy = y - floor(y);
      wz = z - floor(z);
      
      /* bilinear interpolation */
      dens += ((1-wz) * ((1-wy) * *(pos) + wy * *(pos + Mx)) +
	       wz * ((1-wy) * *(pos + Mxy) + wy * *(pos + Mxy_Mx)));
      
      x++;
      y += dydx;
      z += dzdx;      
    }
    dens /= dx; /* correct weighting! */
  }
  else if ((abs_dy >= abs_dx) && (abs_dy >= abs_dz)) //maja 2005
  {
    /* principal direction: y */
    if (y2 < y1)
    {
      /* switch start and stop point */
      dummy = x1; x1 = x2; x2 = dummy;
      dummy = y1; y1 = y2; y2 = dummy;
      dummy = z1; z1 = z2; z2 = dummy;
      
      dx = x2 - x1;
      dy = y2 - y1;
      dz = z2 - z1;
    }
    
    dxdy = dx / dy;
    dzdy = dz / dy;
    
    t = ceil(y1) - y1;
    
    /* (x,y,z) is the location of the first interpolation location */
    y = floor(y1 + t + 0.1);
    x = x1 + t * dxdy;
    z = z1 + t * dzdy;
    
    dens = 0.0;
      
    /* step trough the volume and accumulate the line integral */
     while(y < y2)
    {      
      pos = voxeldata + (int) (floor(z) * Mxy + floor(y) * Mx + floor(x));
      wx = x - floor(x);
      wz = z - floor(z);
      
      /* bilinear interpolation */
       dens += ((1-wz) * ((1-wx) * *(pos) + wx * *(pos + 1)) + 
	       wz * ((1-wx) * *(pos + Mxy) + wx * *(pos + Mxy_1)));
      
      y++;
      x += dxdy;
      z += dzdy;      
    }
    dens /= dy; /* correct weighting! */
  }
  else
    printf("Warning: Joseph line integral only implemented in x- and y-dir.\n");
  
  return dens;
}

/****************************************************************************
 * Line intergration using a variant of Joseph's approach,                  *
 * with nearest neighbor interpolation                                      *
 *                                                                          *
 * author: Henrik Turbell                                            oct 00 *
 * Updated by Maria Magnusson Seger 2005-12-19                              *
 *                                                                          *
 * Input: voxeldata with margin                                             *
 *        size in 3 dimension excluding margin                              *
 *        start and end points given on "unit" cube -1...1                  *
 * Output: none                                                             *
 * Return: average density along the ray                                    *
 ****************************************************************************/
double lineint_in_voxvol_josephnn(float* voxeldata, int Nvox[3],
				  double p1[3], double p2[3])
{
  double x1, x2, y1, y2, z1, z2;
  double dx, dy, dz;
  double abs_dx, abs_dy, abs_dz;  
  int Nx, Ny, Nz, Nxy, Nxyz;
  int Mx, My, Mz, Mxy, Mxy_Mx, Mxy_1;
  double dens = 0.0;
  double t;
  double x, y, z;     
  float* pos;    
  double dummy, dydx, dzdx, dxdy, dzdy;

  /* Sizes without margin */
  /*----------------------*/
  Nx = Nvox[0];
  Ny = Nvox[1];
  Nz = Nvox[2];
  Nxy = Nx * Ny;
  Nxyz = Nxy * Nz;
  
  /* Sizes with margin */
  /*-------------------*/
  Mx = Nx + 2 * VOXMARG;
  My = Ny + 2 * VOXMARG;
  Mz = Nz + 2 * VOXMARG; 
  Mxy = Mx * My;
  Mxy_Mx = Mxy + Mx;
  Mxy_1 = Mxy + 1;
 
  /* Scale from continous unit [-1..1] to    */
  /* discrete voxelindex [0..Nx-1+2*VOXMARG] */
  /* -1 => -0.5 + VOXMARG                    */
  /* 0 => Nx / 2 - 0.5 + VOXMARG             */
  /* 1 => Nx - 0.5 + VOXMARG                 */
  /*-----------------------------------------*/
  x1 = (p1[0] + 1.0) / 2.0 * Nx - 0.5 + VOXMARG;
  x2 = (p2[0] + 1.0) / 2.0 * Nx - 0.5 + VOXMARG;
  y1 = (p1[1] + 1.0) / 2.0 * Ny - 0.5 + VOXMARG;
  y2 = (p2[1] + 1.0) / 2.0 * Ny - 0.5 + VOXMARG;
  z1 = (p1[2] + 1.0) / 2.0 * Nz - 0.5 + VOXMARG;
  z2 = (p2[2] + 1.0) / 2.0 * Nz - 0.5 + VOXMARG;

  dx = x2 - x1;
  dy = y2 - y1;
  dz = z2 - z1; 
  
  abs_dx = fabs(dx);
  abs_dy = fabs(dy);
  abs_dz = fabs(dz); 
  
  /* calculate the line integral either in principal direction x or y */
  /*------------------------------------------------------------------*/
  if ((abs_dx >= abs_dy) && (abs_dx >= abs_dz)) //maja 2005
  {
    /* principal direction: x */
    if (x2 < x1)
    {
      /* switch start and stop point */
      dummy = x1; x1 = x2; x2 = dummy;
      dummy = y1; y1 = y2; y2 = dummy;
      dummy = z1; z1 = z2; z2 = dummy;
      
      dx = x2 - x1;
      dy = y2 - y1;
      dz = z2 - z1;        
    }
           
    dydx = dy / dx;
    dzdx = dz / dx;
    
    t = ceil(x1) - x1;
    
    /* (x,y,z) is the location of the first interpolation location */
    x = floor(x1 + t + 0.1);
    y = y1 + t * dydx;
    z = z1 + t * dzdx;
    
    dens = 0.0;
    
    /* step trough the volume and accumulate the line integral */
    while(x < x2)		
    {      
      pos = voxeldata + (int)(z+0.5) * Mxy + (int)(y+0.5) * Mx + (int)(x+0.5);
      
      /* nearest neighbor interpolation */
      dens += *(pos);
      
      x++;
      y += dydx;
      z += dzdx;      
    }
    dens /= dx; /* correct weighting! */
  }
  else if ((abs_dy >= abs_dx) && (abs_dy >= abs_dz)) //maja 2005
  {
    /* principal direction: y */
    if (y2 < y1)
    {
      /* switch start and stop point */
      dummy = x1; x1 = x2; x2 = dummy;
      dummy = y1; y1 = y2; y2 = dummy;
      dummy = z1; z1 = z2; z2 = dummy;
      
      dx = x2 - x1;
      dy = y2 - y1;
      dz = z2 - z1;
    }
    
    dxdy = dx / dy;
    dzdy = dz / dy;
    
    t = ceil(y1) - y1;
    
    /* (x,y,z) is the location of the first interpolation location */
    y = floor(y1 + t + 0.1);
    x = x1 + t * dxdy;
    z = z1 + t * dzdy;
    
    dens = 0.0;
      
    /* step trough the volume and accumulate the line integral */
     while(y < y2)
    {      
      pos = voxeldata + (int)(z+0.5) * Mxy + (int)(y+0.5) * Mx + (int)(x+0.5);
      
      /* bilinear interpolation */
      dens += *(pos);
      
      y++;
      x += dxdy;
      z += dzdy;      
    }
    dens /= dy; /* correct weighting! */
  }
  else
    printf("Warning: Joseph line integral only implemented in x- and y-dir.\n");
  
  return dens;
}
