/****************************************************************************
 * Some reading functions for polychromatic sources                         *
 *                                                                          *
 * Written by Jens Mueller-Merbach, 1996-03                                 *
 * Updated by Olle Seger, 2003-2005                                         *
 * Updated by Maria Magnusson Seger, 2005-11                                *
 ****************************************************************************/

/*
The program take is licenced under the X11 license:

Copyright (C), 1996-2005, 
Jens Mueller-Merbach, Maria Magnusson, Henrik Turbell, Olle Seger

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>		/* for printf(), ... */
#include <stdlib.h>
#include <errno.h>		/* for errno */
#include <math.h>
#include <malloc.h>		/* for malloc(), realloc(), free() */
#include <string.h>		/* for strstr(), strcpy(), ... */
#include "take.h"

#ifdef STANDALONE //maja 2005-12
#define MALLOC malloc
#define CALLOC calloc
#else
#define MALLOC mxMalloc
#define CALLOC mxCalloc
#endif

#ifndef STANDALONE
#include <mat.h>
#include <mex.h>
#endif

void read_mu_tab(SETUP *su, double *mu, int *mu_lvls);
void read_inspec(double *e_in, SETUP *su, int *e_in_lvls);



/****************************
 * read the material table. *
 ****************************/
void read_mu_tab(SETUP *su,
		 double *mu,	/* energy-attenuation table */
		 int *mu_lvls)	/* number of data pairs in tables */
{
  FILE *fp;
#ifndef STANDALONE
  MATFile *mfp;
  mxArray *pa;
#endif
  char line[STR_LEN];
  int count,i, material, mrows, ncols;
  int pos=0;		       /* starting position in the mu array */
  double erg, Co_CS, In_CS, Ph_CS, *pc;

  for (material=0; material<NR_OF_MAT; material++) {
    if (su->mat_flag[material]) {
      if (strstr(su->mu_file[material],".txt")) { // read .txt-file
	fp = fopen(su->mu_file[material],"rt");
	if (fp==NULL) hejda(su->mu_file[material],4);
	su->mat = 1;
    
	count = 0;
	while(my_fgets(line, fp) != NULL) {
	  if((sscanf(line,"%lf%lf%lf%lf%lf", &erg, &Co_CS, &In_CS, &Ph_CS))!= 4) {
	    fprintf(stderr,"*** Error reading %s.\n", su->mu_file[i]);
	    hejda("Terminated.",9);
	  }

	  mu[pos+count] = 1000.0*erg; /* convert from MeV to keV */
	  if (su->kind_of_att == PHOTO_ATT)
	    mu[pos+count+1] = Ph_CS*100; /* convert from 1/cm to 1/m */
	  else
	    /* Add Coherent, Incoherent, Photoelectric cross sections */
	    mu[pos+count+1] = (Co_CS + In_CS + Ph_CS)*100; 
	  count += 2;
	}
	mu_lvls[material]=count/2;
	pos += count;
	fclose(fp);
      } else if (strstr(su->mu_file[material],".mat")) { // load .mat-file
#ifndef STANDALONE
	    mfp = matOpen(su->mu_file[material], "r");
	    if (mfp == NULL) {
	    hejda(su->mu_file[material], 4);
	    }
	    su->mat = 1;

	    pa = matGetNextVariable(mfp, NULL);
	    mrows = mxGetM(pa);
	    ncols = mxGetN(pa);
	
	    pc= mxGetPr(pa);
	
	    for (i=0; i<mrows; i++) {
	        mu[pos+2*i] = 1000*pc[i]; // convert from MeV to keV
	        if (su->kind_of_att == PHOTO_ATT)
	        mu[pos+2*i+1] = 100*pc[i+mrows]; // convert from 1/cm to 1/m 
	        else
	        /* Add Coherent, Incoherent, Photoelectric cross sections */
	        mu[pos+2*i+1] = 100*(pc[i+mrows]+pc[i+2*mrows]+pc[i+3*mrows]); 
	        }
	        mu_lvls[material]=mrows;
	        pos += 2*mrows;
	        mxDestroyArray(pa);
	    matClose(mfp);
#endif	    
      } else hejda("Terminated.\n", 11);
    }
  } 
}

/****************************************************************************
 * reads the in-spectrum (emitted energy spectrum of the source).           *
 * Note that the first and the last energy intensities must be zero.        *
 ****************************************************************************/
void read_inspec(double *fi, SETUP *su, int *lvls)      
{
  FILE *fp;
  char line[STR_LEN];
  int i, m, n;
  double erg, spec, *pc;
#ifndef STANDALONE
    MATFile *mfp;
    mxArray *pa;
#endif
    
  if (strstr(su->spec_file,".txt")) { // read a .txt-file
    fp = fopen(su->spec_file, "rt");
    if (fp==NULL) hejda(su->spec_file,4);
    su->spect = 1;
    if (su->debug)
      printf("\nLoading energy-spectrum table from\nfile \"%s\".\n", 
	     su->spec_file);
    
    m = 0;
    while(my_fgets(line, fp) != NULL) {
      if((sscanf(line,"%lf%lf", &erg, &spec))!=2) {
	fprintf(stderr,"*** error reading %s.\n", su->spec_file);
	hejda("Terminated.\n",9);
      }
      fi[2*m] = erg;
      fi[2*m+1] = spec;
      m++;
    }
    fclose(fp);
  } else if (strstr(su->spec_file,".mat")) { // load a .mat-file
#ifndef STANDALONE
    mfp = matOpen(su->spec_file, "r");
    if (mfp == NULL) {
      hejda(su->spec_file, 4);
    }
    su->spect = 1;
    
    pa = matGetNextVariable(mfp, NULL);
    if (pa == NULL) {
      hejda("Cannot read matrix!\n", 9);
    }
    m = mxGetM(pa);
    n = mxGetN(pa);
    
    pc= mxGetPr(pa);
    
    for (i=0; i<m; i++) {
      fi[2*i] = pc[i];
      fi[2*i+1] = pc[i+m];
    }
    
    mxDestroyArray(pa);
    matClose(mfp);
#endif
  } else hejda("Terminated.\n", 10);

  if (fi[2*(m-1)] > SMALL_NUMBER) {
    fi[2*m] = 2*fi[2*(m-1)] - fi[2*(m-2)];
    fi[2*m+1] = 0.0;
    m++;
  }

  if (fi[1] > SMALL_NUMBER) {
    printf("m=%d fi[1]=%f\n",m,fi[1]);
    hejda("tjosan",9);
    for (i=m; m>0;i--){		// move the table one row
      fi[2*i]=fi[2*i-2];
      fi[2*i+1]=fi[2*i-1];
    }
    fi[1]=0.0;
    fi[0]=2*fi[2]-fi[4];
  }

  *lvls=m;
}

/****************************************/
/* Read spectrum data and material data */
/****************************************/
void beam_hard(TAKEPAR *pt)
{
  int i,j,l, obj;
  OBJECT *po;
  int e_in_lvls;		/* number of energy levels of e_in */
  double *e_in;			/* emitted energy spectrum: */
				/* e_in[even]: energy level */
				/* e_in[odd]:  intensity */   
				/* e_in[1] = e_in[max] = 0.0 ! */
  double *e_in_w;	        /* interpolated and weighted in-spectrum */
  double sum_in;		/* integral over the in-spectrum */
  int levels;			/* number of energy levels of e_out */
  double *e_out;		/* energy levels where the resulting
				/* beam should be sampled. Must be */
				/* equidistant; calculated by e_in[0], */
				/* e_in[max] and e_out_lvls */
  double *delta_e;		/* energy intervals of the resulting */
				/* spectrum: delta_e[any]=e_out[any+1]-e_out[any] */
  int mu_kinds;			/* number of different materials */
  int *mu_lvls;		        /* number of energy levels of mu[dens] */
  double *mu;			/* energy dependent attenuation: */
				/* mu[even=i] =energy level */
				/* mu[odd=i+1] =attenuation at mu[i] */
  double *mu_i;			/* interpolated mu */
  int elvl_size;	        /* number of energy levels altogether */
  int cnt;			/* counter */
  SETUP *su = pt->psetup;
  DETECTOR *pd = pt->pdetector;
  POLY *pp = su->pp;
 
  /*---------------------------------------------------------------------*
   * Allocate a minimal state and return if kind_of_attenuation = IDEAL  *
   *---------------------------------------------------------------------*/
  if (su->kind_of_att == IDEAL) {
      pp->lvls = 1;

      pp->mu = (double *) CALLOC(1, sizeof(double));
      pp->mu[0] = 1.0;

      pp->nrp = (double *) CALLOC(1, sizeof(double));
      pp->nrp[0] = 1.0;

      pp->mul = (double *) CALLOC(1, sizeof(double));
      pp->mul[0] = 0.0;

      pp->energy = (double *) CALLOC(1, sizeof(double));
      pp->energy[0] = 1.0;

      return;
  }

  /*-----------------------------------------------------------------*
   * Read polychromatic parameters if kind_of_attenuation != IDEAL   *
   *-----------------------------------------------------------------*/
  e_in = (double *) CALLOC(2*MAX_LEVELS, sizeof(double));

  /* read the energy spectrum of the source */
  read_inspec(e_in, su, &e_in_lvls);
  
  /* set the material for each object */
  for (po=su->po; po; po=po->next) {
    su->mat_flag[po->mat] = 1;
  }
      
  /* count the number of different materials/tissues */
  mu_kinds = 0;
  for (i=0; i<NR_OF_MAT; i++) 
    if (su->mat_flag[i])
      mu_kinds = i+1;
  
  su->nr_of_mat = mu_kinds;

  if (su->debug) {
    printf("\nMaterial used\n");
    for (i=0; i<mu_kinds; i++)
      printf("%s %d\n", su->mu_file[i], su->mat_flag[i]);
  }

  /* read the energy-mu-table for each material */
  mu_lvls = (int *) CALLOC(mu_kinds, sizeof(int));
  mu = (double *) CALLOC(NR_OF_MAT*2*MAX_LEVELS, sizeof(double));

  read_mu_tab(su, mu, mu_lvls);

  /*------------------------------------------------------*
   * calculate e_out                                      *
   * calculate delta_energy                               *
   * normalize integral(in-spectrum) to 1                 *
   * calculate interpolation of in_spectrum acc. to e_out *
   * weight w with delta_energy                           *
   * interpolate mu according to e_out                    *
   *------------------------------------------------------*/

  elvl_size = e_in_lvls;

  /* calculate the maximum number of different energy levels */
  for(i=0; i<mu_kinds; i++) {
    if (su->mat_flag[i])
      elvl_size += mu_lvls[i];
  }

  e_out = (double *) MALLOC(elvl_size * sizeof(double));
  cnt = 0;
  for(i=0; i<e_in_lvls; i++) {
    e_out[cnt] = e_in[2*i];
    cnt++;
  }

  l=0;
  for(i=0; i<mu_kinds; i++) {
    if (su->mat_flag[i]) {
      for(j=0; j<mu_lvls[i]; j++) {
	if((mu[l+2*j]>e_in[0])&&(mu[l+2*j]<e_in[2* e_in_lvls-2])) {
	  int mm;
	  e_out[cnt]=mu[l+2*j];
	  cnt++;
	  /* check if energy level already exists */
	  for (mm=0; mm<=(cnt-2); mm++)
	    if(fabs((double)e_out[cnt-1]-e_out[mm]) < SMALL_NUMBER)
	      cnt--;
	}
      }
      l += 2*mu_lvls[i];
    }
  }

  /* all possible energy levels are stored in e_out. 
     Now they have to be sorted -> bubble sort. */
  bubblesort(cnt, e_out);
  levels=cnt;

  /* calculate dEi */
  delta_e=(double *) CALLOC(levels, sizeof(double));
  for(i=1; i<levels-1; i++) {
    delta_e[i]=(e_out[i+1]-e_out[i-1])/2.0;
  }
  delta_e[0]=delta_e[levels-1]=0.0;

  /* interpolate spectrum to all energylevels */
  pp->nrp = e_in_w = (double *) CALLOC(levels, sizeof(double));
  interp(e_in_lvls, e_in, levels, e_out, e_in_w);

  /* normalize spectrum and mult. with dEi */
  sum_in = 0.0;
  for(i=0; i<levels; i++)
    sum_in += delta_e[i]*e_in_w[i];
  for(i=0; i<levels; i++)
    e_in_w[i] *= delta_e[i]/sum_in;

  /* interpolate attenuation curves to all energylevels */
  mu_i = (double *) MALLOC(mu_kinds * levels*sizeof(double));
  l=0;
  for(obj=0; obj<mu_kinds; obj++) {
    interp(mu_lvls[obj], mu+2*l, levels, e_out, mu_i+obj*levels);
    l += mu_lvls[obj];
  }

  pp->mul = (double *) MALLOC(levels * sizeof(double));
  pp->energy = (double *) MALLOC(levels * sizeof(double));

  for (j=0; j< levels; j++) {
    pp->mul[j] = 0.0;
    pp->energy[j] = e_out[j];
    pp->nrp[j] = e_in_w[j]*su->nr_ph*pd->nr_rays;
  }
    
  pp->mu = mu_i;
  pp->lvls = levels;

  /* att. curve for each object */
  for (po=su->po; po; po=po->next) {
    po->mu = pp->mu + pp->lvls*po->mat;
  }
  
  pp->E0 = 0.0;

  // monoenergetic ray?
  if (su->mono > 0.0) {
    for (j=0; j< levels-1; j++) {
      if (pp->energy[j]<su->mono && su->mono <= pp->energy[j+1]) {
	su->mono_i = j+1;
	break;
      }
    }
    if (su->mono_i == 0) hejda("Terminated.", 12);
  }
}     








