%---------------------------------------------
% Reconstruction of a PRESTO-sequence
% 2008-01-30 Written by Per Thyr with help of a
%            program by Marcel Warntjes
% 2008-05-16 Updated by Maria Magnusson 
% 2009-03-11 Updated by Maria Magnusson 
%---------------------------------------------

clear functions

% Correction variables
% --------------------
k1 = 1;		   % 0 = no correction
                   % 1 = Philips phase correction
                   % 2 = Majas phase correction
k1

% special choseable variables
%----------------------------
agar021 = 0;
agar121 = 0;
agar201 = 0;
agar231 = 0;
lundberg_vt10 = 1; 
lundberg_ht10 = 0; 

% Load correction data if nescessary
% ----------------------------------
if (k1==2) load  PHX_images_maja; end;

% Initiate user variables
% -----------------------
doPlotPHXFRX = 0;  % Plot PHX_images? 1: plot, 0: don't plot
iDynamic = 0;      % 0: use 1st dynamic, 1: use 2nd dynamic
ExpMult = 1;       % Multiply with exp() in the z-direction? 1: yes, 0: no

% Look for data-files
% --------------------
[FileName,PathName] = uigetfile('*.mat','Select the MAT-file containing the data', ...
                                '../../fMRI Data/Raw PRESTO/');
if isequal(FileName,0)
    error('User selected Cancel');
else
    disp(['User selected: ', fullfile(PathName, FileName)]);
end
addpath(PathName)
file.name = regexprep(FileName, '.mat', '')
file.list = [file.name '.list'];
file.data = [file.name '.data'];

% Load data files
% ---------------
load([PathName file.name])
load([PathName 'PHX_' file.name])
load([PathName 'FRX_' file.name])
load([PathName 'info_' file.name])

% Opens *.list file and in case of error: display
% -----------------------------------------------
[fidList mesList] = fopen(file.list, 'r'); % Only read (eg. default)
if isempty(mesList) ~= 1
    disp(mesList)
end

% Read information for reconstruction from *.list file
% ----------------------------------------------------
stop_info = 0; % Stop flag
nr_channels = 1;
while stop_info == 0
    list_string = fgetl(fidList); % read one whole row
    parts = strread(list_string, '%s');
    if size(parts,1) > 4
        if strmatch(parts{3}, 'START', 'exact') == 1
            stop_info = 1;
        end
        switch parts{5}
            case 'number_of_encoding_dimensions'
                nr_dims     = str2num(parts{7});
            case 'number_of_dynamic_scans'
                nr_dynamics = str2num(parts{7});
            case 'number'
                if size(parts,1) > 7
                    if strmatch(parts{8}, 'channels', 'exact')
                        nr_channels = str2num(parts{10});
                    end
                end
            case 'kx_range'
                kx_range1 = str2num(parts{7});
                kx_range2 = str2num(parts{8});
            case 'ky_range'
                ky_range1 = str2num(parts{7});
                ky_range2 = str2num(parts{8});
            case 'kz_range'
                kz_range1 = str2num(parts{7});
                kz_range2 = str2num(parts{8});
            case 'kx_oversample_factor'
                kx_oversample = str2num(parts{7});
            case 'ky_oversample_factor'
                ky_oversample = str2num(parts{7});
            case 'kz_oversample_factor'
                kz_oversample = str2num(parts{7});
            case 'number_of_gradient_echoes'
                nr_gradient_echoes = str2num(parts{7});
            case 'X-resolution'
                x_res = str2num(parts{7});
            case 'Y-resolution'
                y_res = str2num(parts{7});
            case 'Z-resolution'
                z_res = str2num(parts{7});
            case 'X_range'
                x_range1 = str2num(parts{7});
                x_range2 = str2num(parts{8});
            case 'Y_range'
                y_range1 = str2num(parts{7});
                y_range2 = str2num(parts{8});
            case 'Z_range'
                z_range1 = str2num(parts{7});
                z_range2 = str2num(parts{8});
        end
    end
end

% resolution in k-space acc to list-file
%---------------------------------------
kx_res = round(x_res*kx_oversample);
ky_res = round(y_res*ky_oversample);
kz_res = round(z_res*kz_oversample);

% Compute midpoints in kspace-data
% --------------------------------
midkx = floor(kx_res/2) + 1;
midky = floor(ky_res/2) + 1;
midkz = floor(kz_res/2) + 1;

% Choose channel number
%----------------------
iChannel  = nr_channels - 1;

kz_start = kz_res*nr_dynamics* iChannel      + iDynamic + 1;
kz_stop  = kz_res*nr_dynamics*(iChannel + 1) + iDynamic;
k_space  = k_images(:,:,kz_start : nr_dynamics : kz_stop);
k_sign   = info.attr.sign(:,kz_start : nr_dynamics : kz_stop);

% Plot k-space before kz-correction
% ---------------------------------
figure(1)
subplot(1,2,1), imagesc(angle(squeeze(k_space(38,:,:))));
%axis image, 
colorbar, title('k_y = 38, Phase'), xlabel('k_z'), ylabel('k_x');
figure(2)
subplot(1,2,1), imagesc(abs(squeeze(k_space(38,:,:)))); 
%axis image, 
colorbar, title('k_y = 38, Abs'), xlabel('k_z'), ylabel('k_x');

% check so that kz_range is odd and center is in the middle
%----------------------------------------------------------
if kz_range1 ~= (kz_range2 * -1)
  disp('Warning! kz_range1 ~= -kz_range2');
end

% compute z_shift and check so that z_resolution is even
%-------------------------------------------------------
z_shift = z_range1 + floor(z_res/2);
if floor(z_res/2)*2 ~= z_res
  disp('Warning! z_resolution is odd');
end;

% kz-correction
% Before reconstruction begins, the kz-slices should be multiplied
% with exp(). Instead, the FOV can be shifted after reconstruction.
% Correction due to T_1 Contrast enhancement.
% -----------------------------------------------------------------
if ExpMult == 1
  kz = kz_range1;
  for iAngle = 1:size(k_space,3)
    k_space(:,:,iAngle) = k_space(:,:,iAngle) * exp(-j*2*pi*kz*z_shift/kz_res);
    kz = kz+1;
  end
end

% Plot k-space after kz-correction
% --------------------------------
figure(1)
subplot(1,2,2), imagesc(angle(squeeze(k_space(38,:,:))));
%axis image, 
colorbar, title('k_y = 38, Phase, Corr'), xlabel('k_z'), ylabel('k_x');
figure(2)
subplot(1,2,2), imagesc(abs(squeeze(k_space(38,:,:)))); 
%axis image, 
colorbar, title('k_y = 38, Abs, Corr'), xlabel('k_z'), ylabel('k_x');

% FFT in kx-direction (dim 2)
% ---------------------------
FDkx = fftshift(fft(ifftshift(k_space, 2), [], 2), 2);

% forward-and-back and frequency correction
% -----------------------------------------
FDkxCorr = zeros(size(FDkx));
FDkxCorr(:,midkx+x_range1:midkx+x_range2,:) = ...
FDkx(:,midkx+x_range1:midkx+x_range2,:);
FDkxCorr(38,midkx+x_range1:midkx+x_range2,:) = ...
    FDkx(38,midkx+x_range1:midkx+x_range2,:);
clear FDkx;
for iZ = 1 : size(k_space,3)
    for iRow = 1 : size(k_space,1)
        if k_sign(iRow,iZ) == -1
            if k1 == 1
                FDkxCorr(iRow, midkx+x_range1:midkx+x_range2, iZ) = ...
                FDkxCorr(iRow, midkx+x_range1:midkx+x_range2, iZ) .* ...
                PHX_images(2*iChannel + 2,:);% .* FRX_images(2,:);%MAJA
             end
            if k1 == 2
                FDkxCorr(iRow, midkx+x_range1:midkx+x_range2, iZ) = ...
                FDkxCorr(iRow, midkx+x_range1:midkx+x_range2, iZ) .* ...
                PHX_images_maja(2*iChannel + 2,:);% .* FRX_images(2,:);%MAJA
             end
        end
        if k_sign(iRow,iZ) == 1
            if k1 ~= 0
	        FDkxCorr(iRow, midkx+x_range1:midkx+x_range2, iZ) = ...
                FDkxCorr(iRow, midkx+x_range1:midkx+x_range2, iZ) .* ...
                1;%FRX_images(1,:);%MAJA
            end
        end
    end
end

% IFFT in kx-direction (dim 2)
% ----------------------------
k_spaceCorr = fftshift(ifft(ifftshift(FDkxCorr, 2), [], 2),2);

% Plot kspace before and after correction
%----------------------------------------
maxv = max(max(max(abs(k_space))));

figure(3)
colormap jet
subplot(2,2,1), imagesc(abs(squeeze(k_space(:,:,18))),[0 0.5*maxv]), ...
    title('before corr, k_z=18'), xlabel('k_x'), ylabel('k_y'), axis image
subplot(2,2,2), imagesc(abs(squeeze(k_spaceCorr(:,:,18))),[0 0.5*maxv]), ...
    title('after corr, k_z=18'), xlabel('k_x'), ylabel('k_y'), axis image
subplot(2,2,3), imagesc(angle(squeeze(k_space(:,:,18)))), ...
    title('before corr, k_z=18'), xlabel('k_x'), ylabel('k_y'), axis image
subplot(2,2,4), imagesc(angle(squeeze(k_spaceCorr(:,:,18)))), ...
    title('after corr, k_z=18'), xlabel('k_x'), ylabel('k_y'), axis image

figure(4)
colormap jet
subplot(2,2,1), imagesc(abs(squeeze(k_space(:,:,21))),[0 0.5*maxv]), ...
    title('before corr, k_z=21'), xlabel('k_x'), ylabel('k_y'), axis image
subplot(2,2,2), imagesc(abs(squeeze(k_spaceCorr(:,:,21))),[0 0.5*maxv]), ...
    title('after corr, k_z=21'), xlabel('k_x'), ylabel('k_y'), axis image
subplot(2,2,3), imagesc(angle(squeeze(k_space(:,:,21)))), ...
    title('before corr, k_z=21'), xlabel('k_x'), ylabel('k_y'), axis image
subplot(2,2,4), imagesc(angle(squeeze(k_spaceCorr(:,:,21)))), ...
    title('after corr, k_z=21'), xlabel('k_x'), ylabel('k_y'), axis image

clear k_space


% 3D FFT
% ------
k_spacePad = zeros(ky_res, kx_res, kz_res);
k_spacePad(midky+ky_range1:midky+ky_range2,:,midkz+kz_range1:midkz+kz_range2) = ...
k_spaceCorr;
%clear k_spaceCorr;
sd = fftshift(fftn(ifftshift(k_spacePad)));

% Plot kspacePad 
%---------------
maxv = max(max(max(abs(k_spacePad))));
figure(5)
colormap jet
subplot(2,2,1), imagesc(abs(squeeze(k_spacePad(39,:,:)))',[0 0.5*maxv]), ...
    title('abs(kspacePad), k_y=39'), ylabel('k_z'), xlabel('k_x') %, axis image
subplot(2,2,2), imagesc(abs(squeeze(k_spacePad(40,:,:)))',[0 0.5*maxv]), ...
    title('abs(kspacePad), k_y=40'), ylabel('k_z'), xlabel('k_x') %, axis image
subplot(2,2,3), imagesc(abs(squeeze(k_spacePad(41,:,:)))',[0 0.5*maxv]), ...
    title('abs(kspacePad), k_y=41'), ylabel('k_z'), xlabel('k_x') %, axis image
subplot(2,2,4), imagesc(abs(squeeze(k_spacePad(42,:,:)))',[0 0.5*maxv]), ...
    title('abs(kspacePad), k_y=42'), ylabel('k_z'), xlabel('k_x') %, axis image
figure(6)
colormap jet
subplot(2,2,1), imagesc(angle(squeeze(k_spacePad(39,:,:)))',[-pi pi]), ...
    title('angle(kspacePad), k_y=39'), ylabel('k_z'), xlabel('k_x') %, axis image
subplot(2,2,2), imagesc(angle(squeeze(k_spacePad(40,:,:)))',[-pi pi]), ...
    title('angle(kspacePad), k_y=40'), ylabel('k_z'), xlabel('k_x') %, axis image
subplot(2,2,3), imagesc(angle(squeeze(k_spacePad(41,:,:)))',[-pi pi]), ...
    title('angle(kspacePad), k_y=41'), ylabel('k_z'), xlabel('k_x') %, axis image
subplot(2,2,4), imagesc(angle(squeeze(k_spacePad(42,:,:)))',[-pi pi]), ...
    title('angle(kspacePad), k_y=42'), ylabel('k_z'), xlabel('k_x') %, axis image

%clear k_spacePad;

% check so that y_range = y_res
%------------------------------
if (y_range2 - y_range1 + 1) ~= y_res
  disp('Warning! y_range is not equal to y_res');
end;

% Cut to FOV in x, y and z direction
% ----------------------------------
if ExpMult ~= 1
  sd = circshift(sd,[0 0 -z_shift]);
end
y_shift = -floor(y_res/2) - y_range1;
sd = circshift(sd,[y_shift 0 0]);
%FOV = sd(:, midkx+x_range1:midkx+x_range2, midkz+z_range1-z_shift:midkz+z_range2-z_shift);
FOV = sd(:, midkx+x_range1:midkx+x_range2, midkz-20:midkz+20);
%FOV = sd(:, midkx+x_range1:midkx+x_range2, midkz-28:midkz+28);
%clear sd;

% Calculate maxvalue for plot
%---------------------------
if agar021 == 1 | agar121 ==1
    maxv = 1.0*max(max(max(abs(FOV))));
elseif agar201 == 1 | agar231 == 1 
    maxv = 11.5*mean(mean(mean(abs(FOV))));
elseif lundberg_vt10 == 1
    maxv = 5.5*mean(mean(mean(abs(FOV))));
elseif lundberg_ht10 == 1
    maxv = 6.5*mean(mean(mean(abs(FOV))));
end

% Plot reconstructed images in three different views
%---------------------------------------------------
figure(8)
colormap jet

subplot(2,3,1), imagesc(squeeze(abs(FOV(:,:,16))),[0 maxv]), ...
    title('Z = 16'), axis image, axis off
subplot(2,3,2), imagesc(squeeze(abs(FOV(:,:,21))),[0 maxv]), ...
    title('Z = 21'), axis image, axis off
subplot(2,3,3), imagesc(squeeze(abs(FOV(:,:,26))),[0 maxv]), ...
    title('Z = 26'), axis image, axis off
subplot(2,6,7), imagesc(squeeze(abs(FOV(:,34,:))),[0 maxv]), ...
    title('X = 34'), axis image, axis off
subplot(2,6,8), imagesc(squeeze(abs(FOV(:,44,:))),[0 maxv]), ...
    title('X = 44'), axis image, axis off
subplot(2,6,9), imagesc(squeeze(abs(FOV(:,54,:))),[0 maxv]), ...
    title('X = 54'), axis image, axis off
subplot(2,6,10), imagesc(squeeze(abs(FOV(31,:,:))),[0 maxv]), ...
    title('Y = 31'), axis image, axis off
subplot(2,6,11), imagesc(squeeze(abs(FOV(41,:,:))),[0 maxv]), ...
    title('Y = 41'), axis image, axis off
subplot(2,6,12), imagesc(squeeze(abs(FOV(51,:,:))),[0 maxv]), ...
    title('Y = 51'), axis image, axis off

% Plot reconstructed images in z-dir
%-----------------------------------
figure(9)
colormap jet
subplot(3,3,1), imagesc(squeeze(abs(FOV(:,:,1))),[0 maxv]), ...
    title('Z = 1'), axis image
subplot(3,3,2), imagesc(squeeze(abs(FOV(:,:,6))),[0 maxv]), ...
    title('Z = 6'), axis image
subplot(3,3,3), imagesc(squeeze(abs(FOV(:,:,11))),[0 maxv]), ...
    title('Z = 11'), axis image
subplot(3,3,4), imagesc(squeeze(abs(FOV(:,:,16))),[0 maxv]), ...
    title('Z = 16'), axis image
subplot(3,3,5), imagesc(squeeze(abs(FOV(:,:,21))),[0 maxv]), ...
    title('Z = 21'), axis image
subplot(3,3,6), imagesc(squeeze(abs(FOV(:,:,26))),[0 maxv]), ...
    title('Z = 26'), axis image
subplot(3,3,7), imagesc(squeeze(abs(FOV(:,:,31))),[0 maxv]), ...
    title('Z = 31'), axis image
subplot(3,3,8), imagesc(squeeze(abs(FOV(:,:,36))),[0 maxv]), ...
    title('Z = 36'), axis image
subplot(3,3,9), imagesc(squeeze(abs(FOV(:,:,41))),[0 maxv]), ...
    title('Z = 41'), axis image

% Plot phase images in z-dir
%---------------------------
figure(10)
colormap jet
subplot(3,3,1), imagesc(squeeze(angle(FOV(:,:,1))),[-pi pi]), ...
    title('Z = 1'), axis image
subplot(3,3,2), imagesc(squeeze(angle(FOV(:,:,6))),[-pi pi]), ...
    title('Z = 6'), axis image
subplot(3,3,3), imagesc(squeeze(angle(FOV(:,:,11))),[-pi pi]), ...
    title('Z = 11'), axis image
subplot(3,3,4), imagesc(squeeze(angle(FOV(:,:,16))),[-pi pi]), ...
    title('Z = 16'), axis image
subplot(3,3,5), imagesc(squeeze(angle(FOV(:,:,21))),[-pi pi]), ...
    title('Z = 21'), axis image
subplot(3,3,6), imagesc(squeeze(angle(FOV(:,:,26))),[-pi pi]), ...
    title('Z = 26'), axis image
subplot(3,3,7), imagesc(squeeze(angle(FOV(:,:,31))),[-pi pi]), ...
    title('Z = 31'), axis image
subplot(3,3,8), imagesc(squeeze(angle(FOV(:,:,36))),[-pi pi]), ...
    title('Z = 36'), axis image
subplot(3,3,9), imagesc(squeeze(angle(FOV(:,:,41))),[-pi pi]), ...
    title('Z = 41'), axis image%, colorbar

% Plot PHX_images
%----------------
if doPlotPHXFRX == 1
    figure(31)
    subplot(2,1,1)
    plot(x_range1:x_range2, angle(PHX_images(1,:)), '-r', ...
         x_range1:x_range2, angle(PHX_images(2,:)), '*:'), ...
    axis([x_range1, x_range2, -pi, pi]); 
    title('PHX-images(2,:)'), grid
    xlabel('sample'), ylabel('angle (rad)')
    if k1==2
        subplot(2,1,2)
        plot(x_range1:x_range2, angle(PHX_images_maja(1,:)), '-r', ...
             x_range1:x_range2, angle(PHX_images_maja(2,:)), '*:'), ...
        axis([x_range1, x_range2, -pi, pi]); 
        title('PHX-images-maja(2,:)'), grid
        xlabel('sample'), ylabel('angle (rad)')
    end
end


